<?php

/**
 * I am THE file that defines the Sessions Manager plugin class
 * The only one.
 *
 * I extend Glb_Pugin, and take profit of all it's functionalities
 *
 * This file is distributed under the same license as the Gloubi_Boulga_Sessions_Manager package.
 *
 * @since             0.0.1
 * @link              
 * @author            John Smith
 * @package           Gloubi_Boulga_Login_Sessions_Manager
 */

class Glb_Sessions extends Glb_Plugin {

    // Define buddies, ie sub-plugins, such as admin / public controllers, and their accessibility
    protected $_buds = [
        [
            'name'          => 'admin',
            'src'           => 'controllers/class-glb-sessions-admin.php',
            'class'         => 'Glb_Sessions_Admin',
            'conditions'    => 'IS_FROM admin WITH_CAPS manage_options'
        ],
    ];

    // Define style files and their accessibility
    protected $_styles = [
        ['handle' => 'glb-sessions-admin', 'src' => 'files/css/glb-sessions-admin.css', 'media' => 'all',
            'conditions' => 'IS_FROM admin WITH_CAPS manage_options'],
        ['handle' => 'glb-sessions-public', 'src' => 'files/css/glb-sessions-public.css', 'media' => 'all',
            'conditions' => 'IS_FROM public]'],
        ['handle' => 'jquery-ui', 'src' => 'vendors/jquery-ui/{{ver}}/themes/smoothness/jquery-ui.css',
            'media' => 'all', 'ver' => '1.11.4'],
    ];

    // Define script files and their accessibility
    protected $_scripts = [
        ['handle' => 'jquery-ui', 'src' => 'vendors/jquery-ui/{{ver}}/jquery-ui.js', 'media' => 'all', 'ver' => '1.11.4'],
        ['handle' => 'glb-sessions-admin', 'src' => 'files/js/glb-sessions-admin.js', 'media' => 'all',
            'deps' => ['jquery'], 'conditions' => 'IS_FROM admin WITH_CAPS manage_options'],
        ['handle' => 'glb-sessions-public', 'src' => 'files/js/glb-sessions-public.js', 'media' => 'all',
            'deps' => ['jquery']],
        ['handle' => 'jquery-ui-dialog', 'src' => 'files/js/glb-sessions-public.js', 'media' => 'all',
            'deps' => ['jquery', 'jquery-ui'], 'conditions' => 'IS_FROM admin WITH_CAPS manage_options'],
    ];

    // Settings can not be defined here, because of translation needs, so just take a look a little below
    protected $_settings = [];

	public function __construct() {
        parent::__construct();

        // here, you can define your settings
        $this->_settings = [
            'log_everything' => [
                'data_type' => 'boolean',
                'label' => __glb('Log actions'),
                'default' => true,
                'template' => 'checkbox',
                'description' => __glb('Want to log user login/logout/lost password/password reset actions ?')
            ],
            'timeout_default' => [
                'data_type' => 'integer',
                'label' => __glb('Default timeout'),
                'default' => 2 * 24 * 60 * 60,
                'template' => 'select',
                'values' => Glb_Time::text_to_seconds([__glbr('1 hour'), __glbr('2 hours'), __glbr('3 hours'), __glbr('4 hours'),
                    __glbr('6 hours'), __glbr('12 hours'), __glbr('18 hours'), __glbr('1 day'), __glbr('2 days'), __glbr('3 days'),
                    __glbr('4 days'), __glbr('5 days'), __glbr('6 days'), __glbr('7 days'), __glbr('10 days'), __glbr('14 days'),
                    __glbr('21 days'), __glbr('28 days'), __glbr('1 month'), __glbr('2 months'), __glbr('3 months'), __glbr('6 months')]),
                'description' => __glb('This is the absolute maximum login session duration (with or without any user action). WP default : 2 days.')
            ],
            'timeout_remember' => [
                'data_type' => 'integer',
                'label' => __glb('Remember timeout'),
                'default' => 14 * 24 * 60 * 60,
                'template' => 'select',
                'values' => Glb_Time::text_to_seconds([__glbr('1 hour'), __glbr('2 hours'), __glbr('3 hours'), __glbr('4 hours'),
                    __glbr('6 hours'), __glbr('12 hours'), __glbr('18 hours'), __glbr('1 day'), __glbr('2 days'), __glbr('3 days'),
                    __glbr('4 days'), __glbr('5 days'), __glbr('6 days'), __glbr('7 days'), __glbr('10 days'), __glbr('14 days'),
                    __glbr('21 days'), __glbr('28 days'), __glbr('1 month'), __glbr('2 months'), __glbr('3 months'), __glbr('6 months')]),
                'description' => __glb('This is the absolute maximum login session duration if the user cheched "remember" at login (with or without any user action). WP default : 14 days.')
            ],
            'timeout_idle' => [
                'data_type' => 'integer',
                'label' => __glb('Idle timeout'),
                'default' => 0,
                'values' => Glb_Time::text_to_seconds([__glbr('Inactive'), __glbr('10 seconds'), __glbr('30 seconds'), __glbr('1 minute'), __glbr('1 hour'), __glbr('2 hours'), __glbr('3 hours'), __glbr('4 hours'),
                    __glbr('6 hours'), __glbr('12 hours'), __glbr('18 hours'), __glbr('1 day'), __glbr('2 days'), __glbr('3 days'),
                    __glbr('4 days'), __glbr('5 days'), __glbr('6 days'), __glbr('7 days'), __glbr('10 days'), __glbr('14 days'),
                    __glbr('21 days'), __glbr('28 days'), __glbr('1 month'), __glbr('2 months'), __glbr('3 months'), __glbr('6 months')]),
                'template' => 'select',
                'description' => __glb('This is the maximum duration a login session can stay alive without any user action. Will be inefficient if greatest than "default timeout".')
            ],
            'timeout_check_admin' => [
                'data_type' => 'integer',
                'label' => __glb('Session check frequency on admin side'),
                'default' => 0,
                'template' => 'select',
                'values' => Glb_Time::text_to_seconds([__glbr('Inactive'), __glbr('15 seconds'), __glbr('20 seconds'),
                        __glbr('25 seconds'), __glbr('30 seconds'), __glbr('30 seconds'), __glbr('35 seconds'), __glbr('40 seconds'),
                        __glbr('45 seconds'), __glbr('50 seconds'), __glbr('55 seconds'), __glbr('60 seconds')]
                ),
                'description' => __glb('On admin side, the frequency of session validity verification.')
            ],
            'timeout_check_front' => [
                'data_type' => 'integer',
                'label' => __glb('Session check frequency on front side'),
                'default' => 0,
                'template' => 'select',
                'values' => Glb_Time::text_to_seconds([__glbr('Inactive'), __glbr('15 seconds'), __glbr('20 seconds'),
                        __glbr('25 seconds'), __glbr('30 seconds'), __glbr('30 seconds'), __glbr('35 seconds'), __glbr('40 seconds'),
                        __glbr('45 seconds'), __glbr('50 seconds'), __glbr('55 seconds'), __glbr('60 seconds')]
                ),
                'description' => __glb('On front side, the frequency of session validity verification.')
            ],
            'timeout_check_front_pages' => [
                'data_type' => 'text',
                'label' => __glb('... Only for pages'),
                'default' => '',
                'template' => 'textarea',
                'template_attrs' => ['rows' => 10, 'cols' => 50],
                'dependency' => 'timeout_check_front',
                'description' => __glb('One page by line. Example : "/" for homepage, or "/example-page/" for example page. If left blank, then all pages will be concerned.')
            ],
        ];

        $this->add_filter('auth_cookie_expiration',     [$this, '_auth_cookie_expiration'], 2, 3);
        $this->add_filter('heartbeat_settings',         [$this, '_heartbeat_settings'], 2, 1);
        $this->add_filter('heartbeat_send',             [$this, '_process_idle'], 2, 2);
        $this->add_action('wp_logout',                  [$this, '_wp_logout'], 2, 2);
        $this->add_action('wp_login',                   [$this, '_wp_login'], 2, 2);
        $this->add_filter('init',                       [$this, '_process_idle'], 2);
        $this->add_action('admin_enqueue_scripts',      [$this, '_admin_enqueue_scripts']);
        $this->add_action('wp_enqueue_scripts',         [$this, '_wp_enqueue_scripts']);
        $this->add_action('password_reset',                   [$this, '_password_reset'], 2, 2);
        $this->add_filter('allow_password_reset',                   [$this, '_allow_password_reset'], 2, 2);

        //$this->add_action('login_enqueue_scripts',      [$this, '_login_enqueue_scripts']);


    }

    function _password_reset($user, $new_pass) {

        if (!$this->settings->get_value('log_everything')) {
            return;
        }
        //Glb_Session::instance()->set('session.user', $user);
        //Glb_Session::instance()->set('session.user_login', $user_login);
        $current_user_id = get_current_user_id();
        if (empty($current_user_id)) { $current_user_id = $user->ID; }
        Glb_Log::notice('_password_reset ' . session_id() . ' : ' . print_r($user, true));
        Glb_Db_Log::instance($this)->log('session_password_reset', GLB_ACTIVITY_TYPE_USER_ACTION,
            __glbr('Password reset'), null, 'info', null, $current_user_id);
    }

    // make sure the stylesheet appears on the lightboxed login iframe
    function _allow_password_reset($allow, $user_id) {
        Glb_Log::notice('_allow_password_reset ' . session_id() . ' : ' . print_r($user_id, true));
        $current_user_id = get_current_user_id();
        if (empty($current_user_id)) { $current_user_id = $user_id; }
        Glb_Db_Log::instance($this)->log('session_password_lost', GLB_ACTIVITY_TYPE_USER_ACTION,
            __glbr('Lost password'), null, 'info', null, $current_user_id);
        return $allow;
    }

    // make sure the stylesheet appears on the lightboxed login iframe
    function _login_enqueue_scripts() {
        wp_enqueue_style( 'wp_auth_check','/wp-includes/css/wp-auth-check.css', array( 'dashicons' ), NULL, 'all' );
    }

    /**
     * Run the plugin, comes just after initialization
     */
    public function run() {
        parent::run();
    }

    /*
     * Check if session is valid
     * If not, silently log out the user
     */
    public function _process_idle($response = [], $data = null) {
        if (!is_user_logged_in()) {
            return $response;
        }
        if (!$this->_check_idle()) {
            // can't call wp_logout because of redirect
            $this->_silent_logout();
            if (is_array($response)) {
                $response['wp-auth-check'] = false;
            }
        }
        return $response;
    }

    /*
     * Hook user logout
     */
    public function _wp_logout($reason = null) {

        if (!$this->settings->get_value('log_everything')) {
            return;
        }

        $message = __glbr('User logged out');
        if ($reason) {
            $message = __glbr('User logged out because of idle timeout');
        }
        $user_id = get_current_user_id();
        if (empty($user_id)) {
            $user_id = Glb_Session::instance()->get('session.user.ID');
        }
        Glb_Db_Log::instance($this)->log('session_logout', GLB_ACTIVITY_TYPE_USER_ACTION, $message);
        //Glb_Log::notice('logged out ' . print_r(Glb_Session::instance()->get('session.user'), true));
        $this->_delete_timestamp();
    }

    public function _wp_login($user_login, $user) {
        if (!$this->settings->get_value('log_everything')) {
            return;
        }
        Glb_Session::instance()->set('session.user', $user);
        Glb_Session::instance()->set('session.user_login', $user_login);
        Glb_Log::notice('logged in ' . session_id() . ' : ' . $user_login);
        Glb_Db_Log::instance($this)->log('session_login', GLB_ACTIVITY_TYPE_USER_ACTION, __glbr('User logged in'));
        $this->_update_timestamp();
    }

    /*
     * Check if user must be logged out because of idle timeout
     * Won't logout the user
     * Test on is_user_logged_in() must be one before
     */
    public function _check_idle() {

        Glb_Log::trace('_check_session_validity ' . $this->settings->get_value('timeout_idle'));

        if (!$this->settings->get_value('timeout_idle')) {
            return true;
        }
        $timestamp = Glb_Session::instance()->get('session.last_action');
        if (time() - $timestamp > $this->settings->get_value('timeout_idle')) {
            return false;
        }
        // update last action only if we are not heartbeating
        ;
        //if ( !(is_admin() && defined( 'DOING_AJAX' ) && DOING_AJAX && $_POST['action'] == 'heartbeat')) {
        if (!$this->request->is('heartbeat')) {
            Glb_Session::instance()->set('session.last_action', time());
        } else {
            Glb_Log::notice('is heartbeat');
        }
        return true;
    }

    /*
     * Configure heartbeat frequency
     */
    public function _heartbeat_settings($settings) {
        if (is_admin()) {
            $frequency = $this->settings->get_value('timeout_check_admin');
        } else {
            $frequency = $this->settings->get_value('timeout_check_front');
        }
        Glb_Log::notice('_heartbeat_settings ' . $frequency . ' ' . $_SERVER['REQUEST_URI']);
        $settings['interval'] = $frequency;
        return $settings;
    }

    protected function _get_timestamp() {
        return Glb_Session::instance()->get('session.last_action', 0);
    }

    protected function _update_timestamp($time = null) {
        return Glb_Session::instance()->set('session.last_action', ( $time === null ? time() : $time));
    }

    protected function _delete_timestamp() {
    }

    /**
     * @function set_auth_timeout
     */
    public function _auth_cookie_expiration($seconds, $user_id, $remember) {
        Glb_Log::trace('_set_auth_timeout ' . $seconds . ' :: ' . $this->settings->get_value('timeout_idle'));
        if ($this->settings->get_value('timeout_idle')) {
            //set_transient('glb_session_' . session_id(), time(), $this->settings->get_value('timeout_idle'));
            $this->_update_timestamp();
        }
        $expiration = ( $remember ? $this->settings->get_value('timeout_remember') : $this->settings->get_value('timeout_default'));
        // http://en.wikipedia.org/wiki/Year_2038_problem
        if ( PHP_INT_MAX - time() < $expiration ) {
            $expiration = PHP_INT_MAX - time() - 5;
        }
        Glb_Log::trace('_set_auth_timeout ' . $expiration);
        return $expiration;
    }

    public function _admin_enqueue_scripts() {
        // don't enqueue for non logged user
        if (!is_user_logged_in()) {
            return;
        }
        Glb_Log::trace('_admin_enqueue_scripts ' . $this->settings->get_value('timeout_check_admin'));
        if (!$this->settings->get_value('timeout_check_admin')) {
            Glb_Log::trace('_admin_enqueue_scripts deregister');
            // don't enqueue if "Session check frequency on admin side" has been checked
            wp_deregister_script( 'heartbeat' );
            wp_deregister_script( 'wp_auth_check' );
        } else {
            // enqueue otherwise
            wp_enqueue_script( 'heartbeat' );
        }
    }

    protected function _silent_logout() {
        $this->_wp_logout('idle');
        wp_destroy_current_session();
        wp_clear_auth_cookie();
        wp_set_current_user( 0 );
    }

    public function _wp_enqueue_scripts() {
        // don't enqueue for non logged user
        if (!is_user_logged_in()) {
            return;
        }
        Glb_Log::notice('_front_enqueue_scripts ' . $this->settings->get_value('timeout_check_front'));

        $do_it  = !empty($this->settings->get_value('timeout_check_front'));
        if ($do_it) {
            $pages = $this->settings->get_value('timeout_check_front_pages');
            $do_it = true;
            if ($pages) {
                $pages = preg_split('/\r\n|\r|\n/', $pages);
                $pages = array_filter($pages, 'trim');
                $pages = array_map(function($var) {
                    return Glb_Text::add_trailing(Glb_Text::add_leading($var, '/'), '/');

                }, $pages);
                $do_it = in_array(Glb_Request::instance()->get_uri(), $pages);
            }
        }
        if (!$do_it) {

            Glb_Log::notice('_wp_enqueue_scripts deregister');
            // don't enqueue if "Session check frequency on admin side" has been checked
            wp_deregister_script( 'heartbeat' );
            wp_deregister_script( 'wp_auth_check' );

        } else {

            Glb_Log::notice('register');

            // enqueue otherwise
            wp_enqueue_script( 'heartbeat' );
            wp_register_script( 'wp_auth_check', '/wp-includes/js/wp-auth-check.js' , array('heartbeat'), false, 1);
            wp_localize_script( 'wp_auth_check', 'authcheckL10n', array(
                'beforeunload' => __glb('Your session has expired. You can log in again from this page or go to the login page.'),
                'interval' => $this->settings->get_value('timeout_check_front'),
            ) );
            wp_enqueue_script ('wp_auth_check');

            // add css file
            wp_enqueue_style( 'wp_auth_check','/wp-includes/css/wp-auth-check.css', array( 'dashicons' ), NULL, 'all' );

            // add the login html to the page
            add_action( 'wp_print_footer_scripts', 'wp_auth_check_html', 5 );
        }


    }

    /**
     * Gloubi Core will dispatch this event as often as it can (hourly by default)
     * @param $args
     */
    public function event_cron($args) {
        Glb_Log::notice('sessions event_cron');
    }

}
