<?php

use PHPUnit\Framework\TestCase;

class Glb_Text_Tests extends Glb_Test_Case {

    public function testDelimit() {
        $result = Glb_Text::delimit('This is my - string WithALot of$$$ Things inside', '$');
        $this->assertEquals($result, 'this$is$my$string$with$a$lot$of$things$inside', '#1 delimit()');
    }

    public function testDasherize() {
        $result = Glb_Text::dasherize('This is my - string WithALot of$#^$$ Things inside');
        $this->assertEquals($result, "this-is-my-string-with-a-lot-of-things-inside", '#1 dasherize()');
    }

    public function testHumanize() {
        $result = Glb_Text::humanize('This is my - string WithALot of$#^$$ Things inside');
        $this->assertEquals($result, "This Is My String With A Lot Of Things Inside", '#1 humanize()');
    }

    public function testCamelize() {
        $result = Glb_Text::camelize('This is my - string WithALot of$#^$$ Things inside');
        $this->assertEquals($result, "ThisIsMyStringWithALotOfThingsInside", '#1 humanize()');
    }

    public function testSlug() {
        $result = Glb_Text::slug('This is my - strïng WithÂLôt of$#^$$ Things inside', '-');
        $this->assertEquals($result, "this-is-my-string-with-a-lot-of-things-inside", '#1 humanize()');
    }

    public function testContains() {
        $result = Glb_Text::contains('$^This is my - strïng WithÂLôt of$#^$$ Things inside', '$^Thi');
        $this->assertEquals($result, true, '#1 contains()');

        $result = Glb_Text::contains('$^This is my - strïng WithÂLôt of$#^$$ Things inside', 'ÂLôt of$#^');
        $this->assertEquals($result, true, '#2 contains()');

        $result = Glb_Text::contains('$^This is my - strïng WithÂLôt  of$#^$$ Things inside', 'ÂLôt of$#^');
        $this->assertEquals($result, false, '#3 contains()');

        $result = Glb_Text::contains('$^This is my - strïng WithÂ Lôt of$#^$$ Things inside', 'XXX');
        $this->assertEquals($result, false, '#4 contains()');
    }

    public function testRemoveTrailing() {

        $result = Glb_Text::remove_trailing('$^This is my - strïng WithÂLôt of$#^$$ Things inside', 'inside');
        $this->assertEquals($result, '$^This is my - strïng WithÂLôt of$#^$$ Things ', '#1 remove_trailing()');

        $result = Glb_Text::remove_trailing('$^This is my - strïng WithÂLôt of$#^$$ Things inside', 'inside ');
        $this->assertEquals($result, '$^This is my - strïng WithÂLôt of$#^$$ Things inside', '#2 remove_trailing()');

        $result = Glb_Text::remove_trailing('$^This is my - strïng WithÂLôt  of$#^$$ Things inside\/', '\/');
        $this->assertEquals($result, '$^This is my - strïng WithÂLôt  of$#^$$ Things inside', '#3 remove_trailing()');

        $result = Glb_Text::remove_trailing('/Remove/Trailing/', '/');
        $this->assertEquals($result, '/Remove/Trailing', '#4 remove_trailing()');

        $result = Glb_Text::remove_trailing('/Remove/Trailing\\', '\\');
        $this->assertEquals($result, '/Remove/Trailing', '#5 remove_trailing()');

    }

    public function testStartsWith() {

        $result = Glb_Text::starts_with('/$^This is', '$^Thi');
        $this->assertEquals($result, false, '#1 starts_with()');

        $result = Glb_Text::starts_with('/$^This is', '/$');
        $this->assertEquals($result, true, '#2 starts_with()');

        $result = Glb_Text::starts_with('/$^This is', '\/$');
        $this->assertEquals($result, false, '#3 starts_with()');

        $result = Glb_Text::starts_with(' This is', 'This is');
        $this->assertEquals($result, false, '#4 starts_with()');

        $result = Glb_Text::starts_with('this is', 'This is');
        $this->assertEquals($result, false, '#5 starts_with()');

        $result = Glb_Text::starts_with('This is me', 'This is');
        $this->assertEquals($result, true, '#6 starts_with()');

    }

    public function testEndsWith() {

        $result = Glb_Text::ends_with('/$^This is', '/$^This is');
        $this->assertEquals($result, true, '#1 ends_with()');

        $result = Glb_Text::ends_with('/$^This is', 'is');
        $this->assertEquals($result, true, '#2 ends_with()');

        $result = Glb_Text::ends_with('^This is*', 'is*');
        $this->assertEquals($result, true, '#3 ends_with()');

        $result = Glb_Text::ends_with(' This is\*', '\*');
        $this->assertEquals($result, true, '#4 ends_with()');

    }

    public function testRemoveLeading() {

        $tests = [
            [ 'remove_leading()', ['Glb_Text', 'remove_leading'], ['This is a test/', '/'], 'This is a test/' ],
            [ 'remove_leading()', ['Glb_Text', 'remove_leading'], ['/This is a test/', '/'], 'This is a test/' ],
            [ 'remove_leading()', ['Glb_Text', 'remove_leading'], ['\/This is a test/', '/'], '\/This is a test/' ],
            [ 'remove_leading()', ['Glb_Text', 'remove_leading'], ['\/This is a test/', '\\'], '/This is a test/' ],
            [ 'remove_leading()', ['Glb_Text', 'remove_leading'], ['\This is a test/', '\\'], 'This is a test/' ],
            [ 'remove_leading()', ['Glb_Text', 'remove_leading'], ['This is a test/', 'This '], 'is a test/' ],
        ];
        $this->_testBulkEquals($tests);

    }

    public function testBetween() {

        $tests = [
            [ 'between()', ['Glb_Text', 'between'], ['{{This is a test}}{{a real test}}', '{{', '}}'], 'This is a test' ],
            [ 'between()', ['Glb_Text', 'between'], ['{{This is a test}}', '{{', '}}'], 'This is a test' ],
            [ 'between()', ['Glb_Text', 'between'], ['This is the test {{#1}}{{#2}}', '{{', '}}'], '#1' ],
            [ 'between()', ['Glb_Text', 'between'], ['This is the test {{#1}}{{{{#2}}', '{{{{', '}}'], '#2' ],
            [ 'between()', ['Glb_Text', 'between'], ['This \\is the test\\ {{#1}}{{{{#2}}', '\\', '\\'], 'is the test' ],
            [ 'between()', ['Glb_Text', 'between'], ['\'test\'', '\'', '\''], 'test' ],
        ];
        $this->_testBulkEquals($tests);
    }

    public function testRemoveBetween() {

        $processed = '{{It\'s a test}}{{a real test}}';
        $sep = ['{{', '}}'];

        $result = Glb_Text::remove_between($processed, $sep[0], $sep[1]);
        $this->assertEquals('It\'s a test', $result, '#1 remove_between(result)');
        $this->assertEquals('{{a real test}}', $processed, '#1 remove_between()');

        $processed = '\'test 1\'\'test 2\' \'test 3\' \'test 4\'';
        $sep = ['\'', '\''];

        $result = Glb_Text::remove_between($processed, $sep[0], $sep[1]);
        $this->assertEquals('test 1', $result, '#2 remove_between(result)');
        $this->assertEquals('\'test 2\' \'test 3\' \'test 4\'', $processed, '#2 remove_between()');

        $processed = 'test.between.points.';
        $sep = ['.', '.'];

        $result = Glb_Text::remove_between($processed, $sep[0], $sep[1]);
        $this->assertEquals('between', $result, '#3 remove_between(result)');
        $this->assertEquals('testpoints.', $processed, '#3 remove_between()');

        $processed = 'test.between..points.';
        $sep = ['.', '.'];

        $result = Glb_Text::remove_between($processed, $sep[0], $sep[1]);
        $this->assertEquals('between', $result, '#4 remove_between(result)');
        $this->assertEquals('test.points.', $processed, '#4 remove_between()');

    }

    public function testBetweenAll() {
        $tests = [
            [ 'between_all()', ['Glb_Text', 'between_all'], ['{{This is a test}}{{a real test}}', '{{', '}}'], ['This is a test', 'a real test'] ],
            [ 'between_all()', ['Glb_Text', 'between_all'], ['{{It\'s a test}}{{a real test}}', '{{', '}}'], ['It\'s a test', 'a real test'] ],
            [ 'between_all()', ['Glb_Text', 'between_all'], ['\'test 1\'\'test 2\' \'test 3\' \'test 4\'', '\'', '\''], ['test 1', 'test 2', 'test 3', 'test 4'] ],
            [ 'between_all()', ['Glb_Text', 'between_all'], ['test.between.points.', '.', '.'], ['between'] ],
            [ 'between_all()', ['Glb_Text', 'between_all'], ['test.between..points.', '.', '.'], ['between', 'points'] ],
            [ 'between_all()', ['Glb_Text', 'between_all'], ['[test][between].points][]', '[', ']'], ['test', 'between', ''] ],
            [ 'between_all()', ['Glb_Text', 'between_all'], ['[[test][between]].points]][]', '[[', ']]'], ['test][between'] ],
        ];
        $this->_testBulkEquals($tests);
    }

    public function testRemoveBetweenAll() {

        $processed = '{{It\'s a test}}{{a real test}}';
        $sep = ['{{', '}}'];

        $result = Glb_Text::remove_between_all($processed, $sep[0], $sep[1]);
        $this->assertEquals(['It\'s a test', 'a real test'], $result, '#1 remove_between_all(result)');
        $this->assertEquals('', $processed, '#1 remove_between_all()');

        $processed = '\'test 1\'test 2\' \'test 3\' \'test 4\'';
        $sep = ['\'', '\''];

        $result = Glb_Text::remove_between_all($processed, $sep[0], $sep[1]);
        $this->assertEquals(['test 1', ' ', ' '], $result, '#2 remove_between_all(result)');
        $this->assertEquals('test 2test 3test 4\'', $processed, '#2 remove_between_all()');

        $processed = 'test.between.points.';
        $sep = ['.', '.'];

        $result = Glb_Text::remove_between_all($processed, $sep[0], $sep[1]);
        $this->assertEquals(['between'], $result, '#3 remove_between_all(result)');
        $this->assertEquals('testpoints.', $processed, '#3 remove_between_all()');

        $processed = 'test.between..points.';
        $sep = ['.', '.'];

        $result = Glb_Text::remove_between_all($processed, $sep[0], $sep[1]);
        $this->assertEquals(['between', 'points'], $result, '#4 remove_between_all(result)');
        $this->assertEquals('test', $processed, '#4 remove_between_all()');

        $processed = '\'[test][between].points][]';
        $sep = ['[', ']'];

        $result = Glb_Text::remove_between_all($processed, $sep[0], $sep[1]);
        $this->assertEquals(['test', 'between', ''], $result, '#5 remove_between_all(result)');
        $this->assertEquals('\'.points]', $processed, '#5 remove_between_all()');

        $processed = '[[test][between]].points]][]';
        $sep = ['[[', ']]'];

        $result = Glb_Text::remove_between_all($processed, $sep[0], $sep[1]);
        $this->assertEquals(['test][between'], $result, '#6 remove_between_all(result)');
        $this->assertEquals('.points]][]', $processed, '#6 remove_between_all()');

    }


    public function testExtractBoolean() {
        $tests = [
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['true'], true ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['false'], false ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], [null], null ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['toto'], null ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], [1], true ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['1'], true ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['y'], true ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['v'], false ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['oui'], true ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['yes'], true ],
            [ 'extract_boolean()', ['Glb_Text', 'extract_boolean'], ['vrai'], true ],
        ];
        $this->_testBulkEquals($tests);
    }

    public function textConcatenate() {

        $tests = [
            [ 'concatenate()', ['Glb_Text', 'concatenate'], [':', 'first', 'second'], 'first:second' ],
            [ 'concatenate()', ['Glb_Text', 'concatenate'], [':', 'first', null, 'second'], 'first:second' ],
            [ 'concatenate()', ['Glb_Text', 'concatenate'], [[':', '::'], 'first', null, 'second', null, 'third', 'first::second::third']],
            [ 'concatenate()', ['Glb_Text', 'concatenate'], [[':', '::'], 'first', 'second', 'third', 'fourth', 'fifth', 'first:second::third::fifth']],
            [ 'concatenate()', ['Glb_Text', 'concatenate'], [null, 'first', 'second', 'third', 'fourth', 'fifth', 'firstsecondthirdfifth']],
        ];
    }

    public function testPluralize() {

        $result = Glb_Text::pluralize('egg');
        $this->assertEquals($result, "eggs", '#1 pluralize()');

        $result = Glb_Text::pluralize('eggs');
        $this->assertEquals($result, "eggs", '#2 pluralize()');

        $result = Glb_Text::pluralize('variety');
        $this->assertEquals($result, "varieties", '#3 pluralize()');

        $result = Glb_Text::pluralize('varieties');
        $this->assertEquals($result, "varieties", '#4 pluralize()');

        $result = Glb_Text::pluralize('species');
        $this->assertEquals($result, "species", '#5 pluralize()');

        $result = Glb_Text::pluralize('specy');
        $this->assertEquals($result, "species", '#6 pluralize()');

        $result = Glb_Text::pluralize('tomato');
        $this->assertEquals($result, "tomatoes", '#7 pluralize()');

        $result = Glb_Text::pluralize('tomatoes');
        $this->assertEquals($result, "tomatoes", '#8 pluralize()');

        $result = Glb_Text::pluralize('houses');
        $this->assertEquals($result, "houses", '#9 pluralize()');

        $result = Glb_Text::pluralize('status');
        $this->assertEquals($result, "statuses", '#10 pluralize()');

        $result = Glb_Text::pluralize('statuses');
        $this->assertEquals($result, "statuses", '#10 pluralize()');

        $result = Glb_Text::pluralize('matrix');
        $this->assertEquals($result, "matrices", '#11 pluralize()');

        $result = Glb_Text::pluralize('matrices');
        $this->assertEquals($result, "matrices", '#12 pluralize()');

        $result = Glb_Text::pluralize('atlas');
        $this->assertEquals($result, "atlases", '#13 pluralize()');

        $result = Glb_Text::pluralize('tooth');
        $this->assertEquals($result, "teeth", '#14 pluralize()');

        $result = Glb_Text::pluralize('weather');
        $this->assertEquals($result, "weather", '#15 pluralize()');

    }



}
