<?php

use PHPUnit\Framework\TestCase;

class Glb_Path_Tests extends Glb_Test_Case {

    public function testListFiles() {

        $root = __DIR__ . DIRECTORY_SEPARATOR . '~path-tests';
        $root_sep = $root . DIRECTORY_SEPARATOR;

        $files = Glb_Path::list_files($root);
        $result = [ $root_sep . 'file1.txt', $root_sep . 'File2.txt', $root_sep . 'file3.php', $root_sep . 'File4.php'];
        sort($files); sort($result);
        $this->assertEquals($files, $result, '#1 list_files()');

        $files = Glb_Path::list_files($root, '*[13].php', true);
        $result = [Glb_Path::normalize($root_sep . 'file3.php', true)];
        foreach(['folder1', 'folder1/f1-sub1', 'folder1/f1-sub2',
                    'Folder2', 'Folder2/f2-sub1', 'Folder2/f2-sub2',
                    'folder3', 'Folder4'] as $folder) {
            $result[] = Glb_Path::normalize($root_sep . $folder . GLB_SEPARATOR . 'file3.php', true);
        }
        $files = array_map(function($item) {
            return Glb_Path::normalize($item, true);
        }, $files);
        sort($files); sort($result);
        $this->assertEquals($files, $result, '#2 list_files()');

    }

    public function testListFolders() {

        $root = __DIR__ . DIRECTORY_SEPARATOR . '~path-tests';
        $root_sep = $root . DIRECTORY_SEPARATOR;

        $files = Glb_Path::list_folders($root, 'f*1', true);
        $result = [ $root_sep . 'folder1', $root_sep . 'folder1' . DIRECTORY_SEPARATOR . 'f1-sub1'];
        sort($files); sort($result);
        $this->assertEquals($files, $result, '#1 list_folders()');

        $files = Glb_Path::list_folders($root, 'f*1', false);
        $result = [ $root_sep . 'folder1' ];
        sort($files); sort($result);
        $this->assertEquals($files, $result, '#2 list_folders()');

        $files = Glb_Path::list_folders($root, 'F*1', false);
        $result = [ ];
        sort($files);
        $this->assertEquals($files, $result, '#3 list_folders()');

        $files = Glb_Path::list_folders($root, '[fF]*', false);
        $result = [ $root_sep . 'folder1', $root_sep . 'Folder2', $root_sep . 'folder3', $root_sep . 'Folder4' ];
        sort($files); sort($result);
        $this->assertEquals($files, $result, '#4 list_folders()');

    }

    public function testConcat() {

        $tests = [
            [ 'concat()', ['Glb_Path', 'concat'], [['key0/key1', 'key2\\key3', 'key4'], ['type' => 'file_system', 'normalize' => true]], 'key0' . GLB_SEPARATOR . 'key1' . GLB_SEPARATOR . 'key2' . GLB_SEPARATOR . 'key3' . GLB_SEPARATOR . 'key4' ],
            [ 'concat()', ['Glb_Path', 'concat'], [['key0/key1', 'key2\\key3', 'key4'], ['type' => 'file_system', 'normalize' => false]], 'key0/key1' . GLB_SEPARATOR . 'key2\\key3' . GLB_SEPARATOR . 'key4' ],
            [ 'concat()', ['Glb_Path', 'concat'], [['key0/key1', 'key2\\key3', 'key4'], ['type' => 'uri', 'normalize' => true]], 'key0/key1/key2/key3/key4' ],
            [ 'concat()', ['Glb_Path', 'concat'], [['key0/key1', 'key2\\key3', 'key4'], ['type' => 'uri', 'normalize' => false]], 'key0/key1/key2\\key3/key4' ],
        ];
        $this->_testBulkEquals($tests);

    }

    public function testIntersect() {

        $tests = [
            [ 'intersect()', ['Glb_Path', 'intersect'], ['/usr/folder1/folder2/folder4/toto.txt', '/usr/bin/folder2/folder4/toto.txt', true], GLB_SEPARATOR . 'usr'],
            [ 'intersect()', ['Glb_Path', 'intersect'], ['/usr/folder1/folder2/folder4/toto.txt', '/usrx/bin/folder2/folder4/toto.txt', true], ''],
            [ 'intersect()', ['Glb_Path', 'intersect'], ['c:\\usr/folder1/folder2/folder4/toto.txt', 'c:\\usr\folder1/folder2/folder4/toto.txt2', true], 'c:' . GLB_SEPARATOR . 'usr' . GLB_SEPARATOR . 'folder1' . GLB_SEPARATOR . 'folder2' . GLB_SEPARATOR . 'folder4'],
            [ 'intersect()', ['Glb_Path', 'intersect'], ['/usr/folder1/folder2/folder4/toto.txt', '/usr/folder1/bin/folder4/toto.txt', false], '/usr/folder1'],
            [ 'intersect()', ['Glb_Path', 'intersect'], ['/usr/folder1\folder2/folder4/toto.txt', '/usr\folder1/folder2/folder4/toto.txt2', false], '/usr/folder1/folder2/folder4'],
        ];
        $this->_testBulkEquals($tests);

    }

    public function testDiff() {

        $tests = [
            [ 'diff()', ['Glb_Path', 'diff'], ['/usr/folder1/folder2/folder4/toto.txt', '/usr/bin/folder2/folder4/toto.txt', true], 'folder1' . GLB_SEPARATOR . 'folder2' . GLB_SEPARATOR . 'folder4' . GLB_SEPARATOR . 'toto.txt'],
            [ 'diff()', ['Glb_Path', 'diff'], ['/usr/folder1/folder2/folder4/toto.txt', '/usr/folder1/folder2/folder4/toto.txt2', true], 'toto.txt2'],
            [ 'diff()', ['Glb_Path', 'diff'], ['/usr/folder1/folder2/folder4/toto.txt', '/usr/bin/folder2/folder4/toto.txt', false], 'folder1/folder2/folder4/toto.txt'],
            [ 'diff()', ['Glb_Path', 'diff'], ['/usr/folder1\folder2/folder4/toto.txt', '/usr\folder1/folder2/folder4/toto.txt2', false], 'toto.txt2'],
        ];
        $this->_testBulkEquals($tests);

    }

}
