<?php

use PHPUnit\Framework\TestCase;

class Glb_Collection_Tests extends TestCase {

    public function testReverseAssociative() {
        $array = ['color1' => 'red', 'color2' => 'green', 'color3' => 'blue'];
        $array_reverse = ['color3' => 'blue', 'color2' => 'green', 'color1' => 'red'];
        $col = new Glb_Collection($array);
        $this->assertEquals($array, $col->to_array(), '#1 to_array()');
        $this->assertEquals($array_reverse, $col->reverse()->to_array(), '#2 reverse()->to_array()');
    }

    public function testReverseMixed() {
        $array = ['color1' => 'red', 'color2' => 'green', 'color3' => 'blue', 'color4 blue'];
        $array_reverse = ['color4 blue', 'color3' => 'blue', 'color2' => 'green', 'color1' => 'red'];
        $col = new Glb_Collection($array);
        $this->assertEquals($array, $col->to_array(), '#1 to_array()');
        $this->assertEquals($array_reverse, $col->reverse()->to_array(), '#2 reverse()->to_array()');
    }

    public function testReverseSequential() {
        $array = ['red', 'green', 'blue', 'white'];
        $array_reverse = ['white', 'blue', 'green', 'red'];
        $col = new Glb_Collection($array);
        $this->assertEquals($array, $col->to_array(), '#1 to_array()');
        $this->assertEquals($array_reverse, $col->reverse()->to_array(), '#2 reverse()->to_array()');
    }

    public function testAddSequential() {

        $array = ['red', 'green', 'blue', 'white'];
        $array_expected = ['red', 'green', 'blue', 'white', 'yellow'];
        $col = new Glb_Collection($array);

        $col->add('yellow');
        $this->assertEquals($array_expected, $col->to_array(), '#1 simple add()');
        $col->remove_value('yellow');
        $this->assertEquals($array, $col->to_array(), '#2 simple remove()');

        $col->add(['key' => 'yellow']);
        $this->assertEquals(['red', 'green', 'blue', 'white', 'key' => 'yellow'], $col->to_array(), '#3 add() with key');
        $col->remove_value('yellow');
        $this->assertEquals(['red', 'green', 'blue', 'white'], $col->to_array(), '#4 add() with key');

        $col->add(['key' => 'yellow']);
        $col->remove_key('key');
        $this->assertEquals(['red', 'green', 'blue', 'white'], $col->to_array(), '#5 add() remove key');

        $col->add(['key' => 'yellow'], 2);
        $this->assertEquals(['red', 'green', 'key' => 'yellow', 'blue', 'white'], $col->to_array(), '#6 simple add() at pos');
        $col->remove_value('yellow');
        $this->assertEquals(['red', 'green', 'blue', 'white'], $col->to_array(), '#7 simple remove() with key at pos');

    }

    public function testCombine() {

        $array = [
            ['id' => 1, 'name' => 'syphilis', 'severity' => '3'],
            ['id' => 2, 'name' => 'cold', 'severity' => '4'],
            ['id' => 3, 'name' => 'cold', 'severity' => '9', 'organ' => 'brain'],
            ['id' => 4, 'name' => 'pneumonia', 'severity' => '2', 'organ' => 'hand'],
            ['id' => 5, 'name' => 'pneumonia', 'severity' => '2', 'organ' => 'foot'],
            ['id' => 6, 'name' => 'pneumonia', 'severity' => '9', 'organ' => 'lung'],
            ['id' => 7, 'name' => 'politics', 'severity' => 'PHP_INT_MAX' ],
        ];
        $col = new Glb_Collection($array);

        $this->assertEquals(json_decode('{"1":"syphilis","2":"cold","3":"cold","4":"pneumonia","5":"pneumonia","6":"pneumonia","7":"politics"}', true),
            $col->combine('id', 'name'), '#1 simple combine()');

        $this->assertEquals(json_decode('{"1":{"syphilis":"3"},"2":{"cold":"4"},"3":{"cold":"9"},"4":{"pneumonia":"2"},"5":{"pneumonia":"2"},"6":{"pneumonia":"9"},"7":{"politics":"PHP_INT_MAX"}}', true),
            $col->combine('id', 'name', 'severity'), '#2 combine() x3 columns');

        $this->assertEquals(json_decode('{"syphilis":{"3":null},"cold":{"4":null,"9":"brain"},"pneumonia":{"2":["hand","foot"],"9":"lung"},"politics":{"PHP_INT_MAX":null}}', true),
            $col->combine('name', 'severity', 'organ'), '#3 combine() x3 columns with merge');

        $this->assertEquals(json_decode('{"syphilis":{"3":null},"cold":{"4":null,"9":"brain"},"pneumonia":{"2":["hand","foot"],"9":"lung"},"politics":{"PHP_INT_MAX":null}}', true),
            $col->combine('name', 'severity', 'organ'), '#3 combine() x3 columns with merge');

        $this->assertEquals(json_decode('{"syphilis-1":"3","cold-2":"4","cold-3":"9","pneumonia-4":"2","pneumonia-5":"2","pneumonia-6":"9","politics-7":"PHP_INT_MAX"}', true),
            $col->combine(function($item) { return $item['name'] . '-' . $item['id']; }, 'severity'), '#4 combine() with anonymous key');

        $this->assertEquals(json_decode('{"syphilis-1":"syphilis-1-3","cold-2":"cold-2-4","cold-3":"cold-3-9","pneumonia-4":"pneumonia-4-2","pneumonia-5":"pneumonia-5-2","pneumonia-6":"pneumonia-6-9","politics-7":"politics-7-PHP_INT_MAX"}', true),
            $col->combine(function($item) { return $item['name'] . '-' . $item['id']; }, function($item) { return $item['name'] . '-' . $item['id'] . '-' .  $item['severity']; }), '#5 combine() with anonymous key/value');

    }

    public function testExtract() {

        $array = [
            ['id' => 1, 'name' => 'syphilis', 'severity' => '3'],
            ['id' => 2, 'name' => 'cold', 'severity' => '4'],
            ['id' => 3, 'name' => 'cold', 'severity' => '9', 'organ' => 'brain'],
            ['id' => 4, 'name' => 'pneumonia', 'severity' => '2', 'organ' => 'hand'],
            ['id' => 5, 'name' => 'pneumonia', 'severity' => '2', 'organ' => 'foot'],
            ['id' => 6, 'name' => 'pneumonia', 'severity' => '9', 'organ' => 'lung'],
            ['id' => 7, 'name' => 'politics', 'severity' => 'PHP_INT_MAX' ],
        ];

        $col = new Glb_Collection($array);
        $this->assertEquals(json_decode('["syphilis","cold","cold","pneumonia","pneumonia","pneumonia","politics"]', true),
            $col->extract('name'), '#1 simple extract()');

        $this->assertEquals(json_decode('[1,2,3,4,5,6,7]', true),
            $col->extract('id'), '#2 simple extract()');

        $this->assertEquals(json_decode('["syphilis-0","cold-1","cold-2","pneumonia-3","pneumonia-4","pneumonia-5","politics-6"]', true),
            $col->extract(function($value, $key) { return $value['name'] . '-' . $key; }), '#3 extract() with anonymous');

    }

    public function testFilter() {

        $array = [
            ['id' => 1, 'name' => 'syphilis     ', 'severity' => '3'],
            ['id' => 2, 'name' => '  cold', 'severity' => '4'],
            ['id' => 3, 'name' => 'cold', 'severity' => '9', 'organ' => 'brain'],
            ['id' => 4, 'name' => 'pneumonia', 'severity' => '2', 'organ' => 'hand'],
            ['id' => 5, 'name' => 'pneumonia', 'severity' => '2', 'organ' => 'foot'],
            ['id' => 6, 'name' => 'pneumonia', 'severity' => '9', 'organ' => 'lung'],
            ['id' => 7, 'name' => 'politics', 'severity' => 'PHP_INT_MAX' ],
        ];

        $col = new Glb_Collection($array);
        $this->assertEquals(json_decode('[{"id":"1","name":"syphilis","severity":"3"},{"id":"2","name":"cold","severity":"4"},{"id":"3","name":"cold","severity":"9","organ":"brain"},{"id":"4","name":"pneumonia","severity":"2","organ":"hand"},{"id":"5","name":"pneumonia","severity":"2","organ":"foot"},{"id":"6","name":"pneumonia","severity":"9","organ":"lung"},{"id":"7","name":"politics","severity":"PHP_INT_MAX"}]', true),
            $col->filter('trim')->to_array(), '#1 filter("trim")');

        $array = [
            ['id' => 1, 'name' => 'syphilis     ', 'severity' => '3'],
            ['id' => 2, 'name' => '  cold', 'severity' => '4'],
            ['id' => 3, 'name' => '', 'severity' => '9', 'organ' => 'brain'],
            ['id' => 4, 'name' => 'pneumonia', 'severity' => '', 'organ' => 'hand'],
            ['id' => 5, 'name' => 'pneumonia', 'severity' => '2', 'organ' => 'foot'],
            ['id' => 6, 'name' => 'pneumonia', 'severity' => '9', 'organ' => 'lung'],
            ['id' => 7, 'name' => 'politics', 'severity' => 'PHP_INT_MAX' ],
        ];
        $col = new Glb_Collection($array);
        $this->assertEquals(json_decode('[{"id":"1","name":"syphilis","severity":"3"},{"id":"2","name":"cold","severity":"4"},{"id":"3","severity":"9","organ":"brain"},{"id":"4","name":"pneumonia","organ":"hand"},{"id":"5","name":"pneumonia","severity":"2","organ":"foot"},{"id":"6","name":"pneumonia","severity":"9","organ":"lung"},{"id":"7","name":"politics","severity":"PHP_INT_MAX"}]', true),
            $col->filter('trim')->to_array(), '#2 filter("trim")');

        $array = [ 'cat     ', 'dog ', '    ', 'mixed', 'key' => ' soap', 'boum' => 'bang' ];
        $col = new Glb_Collection($array);
        $this->assertEquals(json_decode('{"0":"cat","1":"dog","3":"mixed","key":"soap","boum":"bang"}', true),
            $col->filter('trim')->to_array(), '#3 filter("trim")');

        $array = [ 'cat     ', 'dog ', '    ', 'mixed', 'key' => ' soap', 'boum' => 'bang' ];
        $col = new Glb_Collection($array);
        $this->assertEquals(json_decode('{"3":"mixed","boum":"bang"}', true),
            $col->filter(function($value, $key) { return $value == trim($value); }, ARRAY_FILTER_USE_BOTH)->to_array(), '#4 filter(anonymous) both');

        $array = [ 'cat     ', 'dog ', '    ', 'mixed', 'key' => ' soap', 'boum' => 'bang' ];
        $col = new Glb_Collection($array);
        $this->assertEquals(json_decode('["cat     ","dog ","    ","mixed"]', true),
            $col->filter(function($key) { return is_numeric($key); }, ARRAY_FILTER_USE_KEY)->to_array(), '#5 filter(anonymous) key');

        $array = [
            ['id' => 1, 'name' => 'syphilis     ', 'severity' => '3'],
            ['id' => 2, '   name' => '   cold', 'severity' => '4'],
            ['id' => 3, 'name' => 'cold', 'severity' => '9', 'organ' => 'brain'],
            ['id' => 4, 'name' => 'pneumonia ', 'severity' => '2', 'organ ' => 'hand'],
            ['id' => 5, 'name' => 'pneumonia', 'severity' => '2', 'organ' => 'foot'],
            ['id' => 6, 'name' => 'pneumonia ', 'severity' => '9', 'organ' => 'lung'],
            ['id' => 7, 'name' => 'politics', 'severity' => 'PHP_INT_MAX ' ],
        ];
        $col = new Glb_Collection($array);
        $this->assertEquals(json_decode('{"1":{"id":2,"   name":"   cold","severity":"4"},"2":{"id":3,"name":"cold","severity":"9","organ":"brain"},"5":{"id":6,"name":"pneumonia ","severity":"9","organ":"lung"}}', true),
            $col->filter(function($value) {return $value['severity'] > 3; })->to_array(), '#6 filter(anonymous) value');


    }

}
