<?php

/*
 * Glb_Table class.
 */


class Glb_Table
{
    /**
     * @property string $name The Glb_Table name
     *      From the Glb_***_Table definition or by default the sql table name without "glb" nor "wp" prefix
     * @property string $alias The Glb_Table alias
     *      Multiple Glb_Table instances representing the same database table can exist by using different aliases.
     * @property string $sql_name The sql table table name
     * @property string $sql_alias The sql table table name
     */

    /**
     * @var string|null
     */
    protected $_name             = null;

    /**
     * @var null
     */
    protected $_alias            = null;

    /**
     * The table name $table as it was called from Glb_Table::get($table)
     * @var string|null
     */
    protected $_reference               = null;
    protected $_plugin              = null;

    protected $_db                  = null;
    protected $_db_table            = [];

    protected $_behaviors           = [];
    protected $_relations           = [];

    protected $_loaded_relations    = [];
    protected $_loaded_joins        = [];
    //protected $_loaded_matchings    = [];
    protected $_loaded_fields       = [];
    protected $_query_options       = [];

    protected $_options             = [];
    public $entity                  = null;              // the name of entity class

    private $_required              = [];
    protected static $_hooks        = [];

    /**
     * Glb_Table constructor.
     * @param string $table The Table name
     * @param array $options
     */
    function __construct($reference, $options = [])
    {

        //glb_dump($reference  . ' ' . print_r($options, true));
        $this->_reference = $reference;
        $this->_alias = empty($options['alias']) ? Glb_Text::underscore($reference) : $options['alias'];

        if (!empty($options['plugin'])) {
            $this->_plugin = $options['plugin'];
        }

        // find the sql name of the table
        if (empty($options['table'])) {
            $table_name = get_class($this);
            if ($table_name != 'Glb_Table') {
                // the table has a class, we take it
                $table_name = strtolower($table_name);
            } else {
                // use the $reference as table name
                $table_name = strtolower($reference);
            }

            $this->_name = Glb_Text::remove_trailing(Glb_Text::remove_leading($table_name, 'glb_'), '_table');

        } else {
            $this->_name = $options['table'];
        }
        $this->_db = Glb_Db::instance();
        $this->_initialize_db_table(false);

        $this->entity = self::_resolve('entity', $this->reference)['class'];
        $this->_options = $options;
    }

    protected function _initialize_db_table($raise_exception = true) {
        if (empty($this->_db_table)) {
            if (!empty($this->_name)) {
                $this->_db_table = $this->_db->schema()->find_best($this->_name);
            } else {
                $this->_db_table = $this->_db->schema()->find_best($this->_reference);
            }
            if ($raise_exception && empty($this->_db_table)) {
                Throw new Exception('Glb_Table: Unable to find Mysql table ' . Glb_Text::quote_label($this->_name));
            }
        }
    }

    public function get_name($prefixed = true) {
        return $this->_db_table[$prefixed ? 'real_name' : 'name'];
    }

    public function get_alias() {
        return $this->_alias;
    }

    public function __set ( $name , $value ) {
    }

    public function  __get ( $name ) {
        if (in_array($name, ['reference'])) {
            return $this->_reference;
        } else if (in_array($name, ['sql_name'])) {
            return $this->get_name(true);
        } else if (in_array($name, ['sql_alias'])) {
            return $this->get_alias();
        } else if (in_array($name, ['name'])) {
            return $this->_name;
        } else if (in_array($name, ['alias'])) {
            return $this->_alias;
        }
    }

    public function __isset ( $name ) {
        return true;
    }

    public function __unset ( $name ) {
    }

    public function primary_key($index = null) {
        if (!array_key_exists('PRIMARY', $this->_db_table['indexes'])) {
            return null;
        } else {
            if ($index !== null) {
                if (empty($this->_db_table['indexes']['PRIMARY']) ||
                    !is_array($this->_db_table['indexes']['PRIMARY']) ||
                    !array_key_exists($index, $this->_db_table['indexes']['PRIMARY'])) {
                    throw new Exception(sprintf('Glb_Table:: Primary key {0} index not found !'), Glb_Text::quote_label($index));
                }
                return $this->_db_table['indexes']['PRIMARY'][$index];
            } else {
                return $this->_db_table['indexes']['PRIMARY'];
            }
        }
    }

    public function unique_primary_key() {
        if (!array_key_exists('PRIMARY', $this->_db_table['indexes'])) {
            return false;
        } else {
            if (count($this->_db_table['indexes']['PRIMARY']) == 1) {
                return array_values($this->_db_table['indexes']['PRIMARY'])[0];
            } else {
                return false;
            }
        }
    }

    /**
     * To be called in "initialize" function in custom Glb_****_Table class
     * @param string $sql_name The sql table name without WP prefix
     * @param null $sql_alias The alias to be used for this table in future sql queries
     */
    public function set_table($sql_name, $sql_alias = null) {
        $this->_name = $sql_name;
        if (!empty($sql_alias)) {
            $this->_alias = $sql_alias;
        }
        $this->_initialize_db_table(true);
    }

    public function complete_late(&$entity) {
        foreach ($this->_loaded_relations as $alias => $relation) {
            // load late *_to_many
            if (!$relation->is_eager() || ($relation->source == 'matching' && in_array($relation->nature, ['one_to_many', 'many_to_many']))) {
                $entity->archivable(Glb_Entity::ARCHIVABLE_FALSE);
                $entity[$relation->property] = $this->_build_uneager($relation, $entity);
                $entity->archivable(Glb_Entity::ARCHIVABLE_RESTORE);
            }
        }
        return $entity;
    }

    protected function _build_uneager($relation, $entity) {

        // build relation subquery
        if ($relation->nature == 'one_to_many') {

            // build subquery
            $query = $relation->related_table()->query('select')
                ->options($this->_query_options)
                ->fields($this->_required)
                ->where([
                    $relation->related_table()->sql_alias . '.' . $relation->foreign_key => $entity[$relation->binding_key]
                ]);


            $relation->update_query($query);
            $relations = $relation->related_table()->relations(true);
            if (!empty($relations)) {
                $query->contain(array_keys($relations));
            }

            return $query->execute();

        } else if ($relation->nature == 'many_to_many') {

            // build subquery (also)
            //$relation->related_table();
            //$relation->through_table();
            $conditions = array_merge([
                Glb_Query::quote_field($relation->through_table()->sql_alias) . '.' . Glb_Query::quote_field(array_values($relation->binding_key)[0])  . ' = ' . $entity[array_keys($relation->binding_key)[0]],
                Glb_Query::quote_field($relation->through_table()->sql_alias) . '.' . Glb_Query::quote_field(array_keys($relation->foreign_key)[0]) . ' = ' . Glb_Query::quote_field($relation->related_table()->sql_alias) . '.' . Glb_Query::quote_field(array_values($relation->foreign_key)[0])
            ], $relation->conditions());

            $query = $relation->related_table()->query('select')
                ->options($this->_query_options)
                ->fields($this->_required)
                ->join([
                    'alias' => $relation->through_table()->sql_alias,
                    'table' => $relation->through_table()->reference,
                    'conditions' => $conditions,
                    'type' => 'inner',
                ]);

            $relation->update_query($query);
            return $query->execute();

        }

    }

    public function build_entity($row_data, $check_null = false) {

        if (empty($row_data)) { return null; }

        // build local entity properties
        $properties = [];
        foreach($this->_loaded_fields as $field_alias => $field_value) {
            $properties[$field_value['column_alias']] = $row_data[$field_alias];
        }

        // if primary key is null then return false
        if ($check_null) {
            $all_null = true;
            /*$primary_key = $this->primary_key();
            foreach($primary_key as $column) {
                if ( ! ( ($column['column_name']) === null ||
                    $properties[$column['column_name']] === $this->column($column['column_name'])->default) ) {
                    $all_null = false;
                    break;
                }
            }*/

            // can't use primary key because it may not be
            foreach ($properties as $key => $value) {
                if ($value !== null) {
                    $all_null = false;
                    break;
                }
            }
            if ($all_null) { return null; }
        }

        //glb_dump($this);

        /*global $logTable;

        if (!empty($logTable)) {
            glb_dump($this);
            glb_dump(Glb_Array::nice_stack_trace());
        }*/
        foreach ($this->_loaded_relations as $alias => $relation) {
            if ($relation->is_eager()) {
                if ($relation->source == 'matching') {
                    Glb_Array::ensure_key($properties, '_matching_data', []);
                    //$properties['_matching_data'][$relation->property] = $row_data;
                    $properties['_matching_data'][$relation->property] = $relation->related_table()->build_entity($row_data, true);
                    //$relation->related_table()->build_entity($row_data, true);
                } else {
                    $properties[$relation->property] = $relation->related_table()->build_entity($row_data, true);
                }
            }
        }

        if (!empty($this->_loaded_joins)) {
            $properties['_join_data'] = [];
            foreach ($this->_loaded_joins as $alias => $join) {
                $properties['_join_data'][$join['property']] = $join['table_object']->build_entity($row_data, true);
                //$properties[$join['property']] = $join['table_object']->build_entity($row_data, true);
                //$properties[$join['property']] = $join['table_object']->build_entity($row_data, true);
            }
        }

        /*if (!empty($this->_loaded_matchings)) {
            $properties['_matching_data'] = [];
            foreach ($this->_loaded_matchings as $alias => $relation) {
                $properties['_matching_data'][$relation->property] = $relation->related_table()->build_entity($row_data, true);
            }
        }*/


        //$primary_key = $this->primary_key();
        return new $this->entity($this, $properties, false);

    }

    public function build_entities($data, $check_null = false) {
        $collection = [];
        if(empty($data)) { return new Glb_Collection([]); }
        foreach($data as $row_data) {
            $collection[] = $this->build_entity($row_data, $check_null);
        }
        return new Glb_Collection($collection);
    }

    public function get_entity($id, $options = []) {
        $primary_key = $this->unique_primary_key();
        if (!$primary_key) {
            throw new Exception('Glb_Table::get_entity() can only work if sql table have a unique primary key');
        }
        $primary_key = $primary_key['column_name'];
        $query = $this->select()
            ->where([Glb_Query::quote_field($this->sql_alias) . '.' . Glb_Query::quote_field($primary_key) => $id]);

        if ($options) {
            foreach($options as $option_key => $option_value) {
                if (method_exists($query, $option_key)) {
                    $query->{$option_key}($option_value);
                }
            }
        }
        return $query->first();
    }

    public function new_entity($data = null, $options = []) {
        $entity = [];
        Glb_Array::ensure($options);
        foreach($this->columns() as $column) {
            if (!empty($data[$column->name])) {
                $entity[$column->name] = $data[$column->name];
            } else {
                $entity[$column->name] = $column->default;
            }
        }
        return new $this->entity($this, $data, true, $options);
    }

    public function new_entities($data, $options = []) {
        $result = [];
        foreach($data as $row_data) {
            $result[] = $this->new_entity($row_data, $options);
        }
        return $result;
    }

    /**
     * @param $entity
     * @param $data
     * @param array $options ['associate' => ['dependent_table1', 'dependent_table2']]
     * @return mixed
     */
    public function patch_entity($entity, $data, $options = []) {
        $entity = $entity->patch($data, $options);

        // check if some properties have not been set
        if (!empty($options['associate'])) {
            Glb_Array::ensure($options['associate']);
            //$this->compile_relations($options['contain']);
            // @todo : specify fields
            // @todo : check unique primary key
            //$this->compile_fields('*');
            foreach($options['associate'] as $associated) {
                $relation = $this->relation($associated);
                if (array_key_exists($relation->property, $data)) {
                    if (in_array($relation->nature, ['one_to_many'])) {
                        $entities = [];
                        foreach($data[$relation->property] as $item) {
                            $new = $relation->related_table()->new_entity($item);
                            if ($entity->exists($relation->binding_key)) {
                                $new[$relation->foreign_key] = $entity[$relation->binding_key];
                            }
                            $entities[] = $new->reset_new()->reset_archives();
                        }
                        $entity[$relation->property] = new Glb_Collection($entities);
                    } else if (in_array($relation->nature, ['one_to_one', 'many_to_one'])) {
                        $new = $relation->primary_table()->new_entity($data[$relation->property])->reset_new();
                        if ($entity->exists($relation->binding_key)) {
                            $new[$relation->foreign_key] = $entity[$relation->binding_key];
                        }
                        $entity[$relation->property] = $new->reset_new()->reset_archives();
                    } else {
                        throw new Exception('Glb_Table:patch_entity:: Patch with %s option activated can not work on %s relations.', Glb_Text::quote_label('contain'), Glb_Text::quote_label($relation->nature));
                    }
                }
            }
        }
        return $entity->reset_archives();
    }

    /*
     *
     *
     * $options = [
     *      'ignore_callbacks' => [
     *          'tables' => ['articles_categories'], 'behaviors' => ['timestamp']
     *      ],
     *      'associated' => [
     *          'articles_categories',
     *          'articles_tags'
     *      ]
     * ]
     */

    /**
     * Save an entity to its base table : insert if new, update if not
     * @param Glb_Table_Entity $entity
     * @param array $options
     * The options array accepts the following keys:
     *  - skip_validation (false by default) : @todo
     *          true to skip validation rules (according to database fields definitions) / false to apply them
     *  - associate :
     *          false (by default) to avoid saving dependent tables data
     *          array of relations to save at the same time. Only first level relations are associated to the save action.
     */
    public function save_entity(&$entity, $options = []) {

        $values = $fields = [];
        $new = $entity->is_new();

        if ($new) {

            // if new, insert
            $query = $this->query('insert');

        } else {

            // if not new, update
            $query = $this->query('update');
            $primary_key = $this->primary_key();

            // check primary key
            if (empty($primary_key)) {
                throw new Exception('Glb_Table::save_entity() can only work if the sql table have a primary key.');
            }

            // add primary key where conditions
            foreach($primary_key as $field) {
                if (!array_key_exists($field['column_name'], $entity)) {
                    throw new Exception(sprintf('Glb_Table::save_entity() can only work if the primary key is set. %s not set for %s', $field['column_name'], print_r($entity, true)));
                }
                $query->where([$field['column_name'] => $entity[$field['column_name']]]);
            }

        }

        $auto_increment = false;

        // load fields, values, convert data and read autoincrement field
        foreach($this->columns() as $column) {
            if ($column->auto_increment && $column->primary) {
                $auto_increment = $column->name;
            }
            // only filled columns
            if ($entity->exists($column->name)) {
                $fields[] = $column->name;
                $values[] = $column->convert_to($entity[$column->name]);
            }
        }

        // set calculated fields values
        $query->fields($fields)->values($values);
        glb_dump($values);
        glb_dump($query->sql());


        // process current table query
        $exception = '';
        try {
            $id = $query->execute();
            if ($new) {
                // set auto-increment field
                if (is_numeric($id) && !empty($auto_increment)) {
                    $entity->{$auto_increment} = $id;
                }
            }
        } catch(Exception $ex) {
            $exception = 'An exception occurred while inserting : ' . $ex->getMessage() . ' -- for query ' . Glb_Text::quote_label($query->sql());
        }

        // read / set possible errors / warnings
        $last_warnings = $query->last_warnings();
        $last_errors = $query->last_errors();

        foreach($last_errors as $last_error_key => $last_error) {
            $last_errors[$last_error_key] .= ' -- for query ' . Glb_Text::quote_label($query->sql());
        }
        if (!empty($exception)) {
            $last_errors[] = $exception;
        }

        $entity->set_errors($last_errors)->set_warnings($last_warnings)->reset_new();

        // process "associated" relations queries
        if (!empty($options['associate'])) {

            if (!($primary = $this->unique_primary_key())) {
                throw new Exception(sprintf('Glb_Table::save_entity() with %s option activated can only work if the primary key on the main table is set on a single field.', Glb_Text::quote_label('associate')));
            }

            Glb_Array::ensure($options['associate']);
            Glb_Array::convert_associative($options['associate'], []);
            $this->compile_relations($options['associate']);
XXXXXXXXXXXXXXXXX
            // loop through associations
            foreach ($this->_loaded_relations as $relation_key => $relation) {
                if (empty($entity[$relation->property])) {
                    // delete from related_table
                    if (in_array($relation->nature, ['one_to_many', 'one_to_one'])) {
                        // simple delete
                        $relation->related_table()->delete([$relation->foreign_key => $entity[$relation->binding_key]]);
                    } else if (in_array($relation->nature, ['many_to_one'])) {
                        // check if related entity is not still used in the prlmary table

                    } else if (in_array($relation->nature, ['many_to_many'])) {

                    }
                }
            }
        }
/*
            foreach($options['associate'] as $associated_key => $associated) {

                $associated_items = explode('.', $associated_key);
                $current_table = $this; $current_entity = $entity;
                $current_path = '';

                // loop through
                foreach($associated_items as $associated_item) {
                    $current_relation = $current_table->relation($associated_item);

                    // read relations properties
                    $primary_value = $current_entity[$current_relation->binding_key];

                    if ($current_path == '') {
                        $current_path = $current_relation->property;
                    } else {
                        $current_path .= '.' . $current_relation->property;
                    }
                    glb_dump('relation property : '  . $current_relation->property);
                    if ( !$current_entity->exists($current_relation->property) || empty($current_entity[$current_relation->property]) ) {
                        $current_entity[$current_relation->property] = new Glb_Collection();
                    }

                    // find table and keys
                    if ($current_relation->nature == 'many_to_one') {
                        $target_table = $current_relation->through_table();
                        $binding_key = array_keys($current_relation->binding_key)[0];
                        $foreign_key = array_values($current_relation->binding_key)[0];
                    } else {
                        $target_table = $current_relation->related_table();
                        $binding_key = $current_relation->binding_key;
                        $foreign_key = $current_relation->foreign_key;
                    }

                    // ensure entities



                }*/




                /*foreach($entity[$relation->property] as $item) {

                }

                    // remove all values on the join table if expected values of $entity are empty
                    $relation_query = $relation->related_table()->query('delete');
                    $relation_query->where([$relation->foreign_key => $primary_value]);
                    glb_dump('remove all because empty');
                    glb_dump($relation_query->sql());
                    $entity[$relation->property] = null;

                } else {

                    glb_dump('relation property values : ', $entity[$relation->property]);

                    //$relation = new Glb_Relation();
                    if (in_array($relation->nature, ['one_to_many'])) {


                        $exclude_deletions = [];

                        // retrieve entity
                        foreach($entity[$relation->property] as $item) {

                            $exclude_deletions[] =
                            if ($item->is_new()) {
                                // insert

                            } else {
                                // update
                            }

                            // force primary key value
                            $item[$relation->foreign_key] = $primary_value;
                            // find entity
                            $entity_query = $relation->related_table()->query('select');
                            foreach($update_keys as $update_key) {
                                $entity_query->where([GLb_Query::quote_field($relation->related_table()->get_alias()) .
                                    '.' . GLb_Query::quote_field($update_key) => $item[$update_key]]);
                            }
                            $entity_found = $entity_query->first();
                            glb_dump('find_entity before update ' . $entity_query->sql());
                            glb_dump('find_entity before update object ', $entity_found);
                            if ($entity_found) {
                                // update
                            } else {
                                // insert
                            }
                        }
                    }
                }

                glb_dump('processing $relation ', $relation);
            }
        }*/


        //@todo : save associated :
            // load relations
            // find relations
        return $entity;
    }

    public function save_entities($entities, $options = []) {
        $result = [];
        foreach($entities as $entity) {
            $result[] = $this->save_entity($entity, $options);
        }
        return $result;
    }

    public function delete($where) {
        if (empty($entity) || !is_array($where)) {
            return false;
        }
        $query = $this->query('delete');
        $query->where($where);
        return $query->execute();
    }


    public function delete_entity($entity, $options = []) {

        if (empty($entity)) {
            return false;
        }

        $query = $this->query('delete');
        if (is_int($entity)) {
            if (!$this->unique_primary_key()) {
                throw new Exception(
                    sprintf(
                        'Glb_Table::delete_entity() was unable to complete because parameter $entity is integer and primary_key is not unique for table %s.',
                        Glb_Text::quote_label($this->get_alias())
                    )
                );
            }
            $query->where([$this->primary_key(0)['column_name'] => $entity]);
        } else {
            $primary_key = $this->primary_key();
            foreach($primary_key as $field) {
                if (empty($entity[$field['column_name']])) {
                    throw new Exception(
                        sprintf(
                            'Glb_Table::delete_entity() was unable to complete because primary key %s is not set for table %s.',
                            Glb_Text::quote_label($field['column_name']), Glb_Text::quote_label($this->get_alias())
                        )
                    );
                }
                $query->where([$field['column_name'] => $entity[$field['column_name']]]);
            }
        }

        return $query->execute();
    }

    public function delete_entities($entities, $options = []) {
        $result = [];
        foreach($entities as $entity) {
            $result[] = $this->delete_entity($entity, $options);
        }
        return $result;
    }


    /**
     * @param string $by One of 'type', 'property'
     * @param array $values = ['many_to_one', 'one_to_one']
     * @param mixed $loaded 'flat', 'nested', false
     * @return array
     */
    /*public function &get_relations_by($by = 'type', $values = [], $loaded = false) {
        $result = [];
        if ($loaded === true) { $loaded = 'nested'; }
        if ($loaded) {
            $relations = $this->_loaded_relations[$loaded];
        } else {
            $relations = $this->_relations;
        }

        foreach($relations as $relation_key => $relation) {
            if ($by == 'type' && in_array($relation->type, $values)) {
                $result[$relation_key] = $relation;
            } else if ($by == 'property' && in_array($relation->property, $values)) {
                $result[$relation_key] = $relation;
            }
        }
        return $result;
    }*/

    /**
     * @param string $by One of 'type', 'property'
     * @param array $values = example ['many_to_one', 'one_to_one'] or the array of properties to search for
     * @param boolean $loaded Want to search in loaded properties ?
     * @param boolean|string $recursive False : doesn't search recursively, 'flat' or 'nested' : search recursively and return as flat or nested
     * @return array
     */
    public function relations_by($by, $values = [], $loaded = true, $recursive = true, $return = 'flat')
    {

        $result = [];
        if ($loaded) {
            $relations = $this->_loaded_relations;
        } else {
            $relations = $this->_relations;
        }

        Glb_Array::ensure($values);

        //glb_dump('relations_by ' . $by);
        //glb_dump('$relations ', $this->_loaded_matchings);
        foreach ($relations as $relation_key => $relation) {

            //Glb_Array::find_value($relation->$by, $values, 'icase') !== false
            $do_it = false;
            //glb_dump($relation_key);

            if (in_array($by, ['nature', 'property', 'type', 'strategy', 'source']) && (empty($values) || in_array($relation->$by, $values))) {
                $do_it = true;
            } /*else if (in_array($by, ['eagerness', 'eager'])) {
                //if ($values && ( $relation->type == 'inner' || in_array($relation->nature, ['many_to_one', 'one_to_one']) )) {
                if ($values && $relation->is_eager()) {
                    $do_it = true;
                } else if (!$values && !$relation->is_eager()) {
                    $do_it = true;
                }
            }*/
            //glb_dump('$do_it ' . $do_it);

            // first level
            if ($do_it) {

                if ($return == 'flat') {
                    $result[$relation_key] = $relation;
                } else {
                    $result[$relation_key] = ['relation' => $relation, 'children' => []];
                }

                // sub levels
                if ($recursive) {
                    $sub_relations = $relation->related_table()->relations_by($by, $values, $loaded, $recursive, $return);
                    if ($return == 'flat') {
                        foreach ($sub_relations as $sub_relations_key => $sub_relation) {
                            $result[$sub_relations_key] = $sub_relation;
                        }
                    } else {
                        //Glb_Hash::ensure_values($result[$relation_key], ['children'], [[]]);
                        $result[$relation_key]['children'] = $sub_relations;
                    }
                }
            }
        }
        return $result;
    }


    /**
     * Return one relation defined in the table, search only first level relation, no hierarchy
     * @param $alias
     * @param string $loaded 'flat', 'nested', false
     * @return mixed
     */
    public function &relation($alias, $loaded = false) {
        //$aliases = explode('.', $alias);
        $search_in = ($loaded ? $this->_loaded_relations : $this->_relations);

        /*$aliases = explode('.', $alias);
        $current_table = $this;
        $current_relation = null;
        foreach($aliases as $alias_item) {
            $current_relation = $current_table->relation($alias_item, $loaded);
            $current_table = $this;
        }*/

        //echo '<pre>';
        //glb_dump($alias);
        //glb_dump($this->_relations);
        //glb_dump(Glb_Array::nice_stack_trace());
        //echo '</pre>';
        if (!array_key_exists($alias, $search_in)) {
            throw new Exception('Relation ' . Glb_Text::quote_label($alias) . ' not found in ' . __CLASS__ . ' ' . Glb_Text::quote_label($this->sql_alias));
        }
        return $search_in[$alias];
    }

    /**
     * Return relations defined in the table, only first level relations, no hierarchy
     * @param boolean $loaded Boolean true | false, if true, search in loaded relations
     * @return mixed
     */
    public function &relations($loaded = false) {
        if ($loaded) {
            return $this->_loaded_relations;
        } else {
            return $this->_relations;
        }
    }

    /**
     * @param array $relations The relations loaded by query
     *      [
     *          'relation1_alias' => ['source' => 'contain'],
     *          'relation1_alias' => ['source' => 'matching', 'conditions' => []],
     *          ...
     *      ]
     * @param array $query_options
     * @return $this
     * @throws Exception
     */
    public function compile_relations($relations, $query_options = []) {

        $this->_query_options = $query_options;
        $this->_loaded_relations = [];
        if (empty($relations)) { return $this; }

        foreach ($relations as $alias => $item) {

            $relation = $this->relation($alias);
            $relation->compile_config($item);
            if ($item['source'] == 'matching') {
                //$relation->alias = Glb_Text::ensure_leading($relation->alias, '_matching_');
            }

            // compile sub tables
            // @todo test matching level2
            if (!empty($item['contain'])) {
                $relation->related_table()->compile_relations($item['contain'], $this->_query_options);
            }
            $this->_loaded_relations[$alias] = $relation;

        }

        return $this;

    }

    /*
     * $return 'this' | 'flat' | 'nested' | 'both'
     */
    /*public function compile_contain($contains, $query_options = []) {

        $this->_query_options = $query_options;
        $this->_loaded_relations = [];

        if (empty($contains)) {
            return $this;
        }
        //Glb_Log::info('compile_contain item ', $this->alias, $contains);

        foreach ($contains as $alias => $item) {

            // get relation as defined in table definition
            $relation = $this->relation($alias);
            $relation->compile_config($item);
            $relation->source = 'contain';

            // compile sub tables
            // @todo test contain
            if (!empty($item['contain'])) {
                $relation->related_table()->compile_contain($item['contain'], $this->_query_options);
            }
            $this->_loaded_relations[$alias] = $relation;

        }

        return $this;

    }*/

    /*public function compile_matching($matchings, $query_options = []) {

        $this->_query_options = $query_options;
        $this->_loaded_matchings = [];

        if(empty($matchings)) { return $this; }

        foreach ($matchings as $alias => $item) {

            // get relation as defined in table definition
            // clone it so that we can call contain on non eager relations
            $relation = clone $this->relation($alias);
            $relation->compile_config($item);
            $relation->type = 'inner';
            $relation->source = 'matching';
            // compile sub tables
            $this->_loaded_matchings[$alias] = $relation;

        }
        //Glb_Log::info('compile_contain itemX ', $this->alias, $this->_loaded_relations);

        return $this;

    }

    public function compile_not_matching($not_matchings, $query_options = []) {

        $this->_query_options = $query_options;
        //$this->_loaded_matchings = [];
        //glb_dump('compile_unmatching', $unmatchings);

        if(empty($not_matchings)) { return $this; }
        foreach ($not_matchings as $alias => $item) {

            // get relation as defined in table definition
            // clone it so that we can call contain on non eager relations
            $relation = clone $this->relation($alias);
            $relation->compile_config($item);
            $relation->type = 'left';
            $relation->source = 'not_matching';
            // compile sub tables
            $this->_loaded_matchings[$alias] = $relation;

        }
        //Glb_Log::info('compile_contain itemX ', $this->alias, $this->_loaded_relations);
        //glb_dump($this->_loaded_matchings);
        return $this;

    }*/

    /*
     *
     *
     */
    /*public function compile_contain($contains, $query_options = []) {

        $this->_query_options = $query_options;
        $this->_loaded_relations = [];
        $result = [];

        Glb_Log::info(' talbe compile_contain', $contains);
        if(empty($contains)) {
            return $this;
        }
        $contains = Glb_Array::convert_associative($contains);

        foreach ($contains as $alias => $item) {
            //Glb_Log::info('compile_contain item ', $this->alias, $alias);
            $aliases = explode('.', $alias);
            $current_alias = array_shift($aliases);

            // get relation as defined in table definition
            $relation = $this->relation($current_alias);

            // do not update relation exec_config if already exists or alias is composed
            if (!count($aliases) || !array_key_exists($current_alias, $this->_loaded_relations)) {
                $relation->compile_config( count($aliases) ? [] : $item );
            }
            // compile sub tables
            if (count($aliases)) {
                $relation->related_table()->compile_contain([implode('.', $aliases) => $item], $this->_query_options);
            }
            $this->_loaded_relations[$current_alias] = $relation;

        }
        //Glb_Log::info('compile_contain itemX ', $this->alias, $this->_loaded_relations);

        return $this;

    }*/

    protected function _normalize_join($join) {
        if (empty($join['table'])) {
            throw new Exception('Glb_Query : Join ' . Glb_Text::quote_label('table') . ' attribute is required. ' . print_r($join, true));
        }
        $join = Glb_Hash::ensure_values($join, ['type', 'alias', 'conditions'], ['inner', $join['table'], []]);
        if (empty($join['property'])) { $join['property'] = $join['alias']; }

        $join['type'] = strtolower(trim($join['type']));

        if (!in_array($join['type'], ['inner', 'left', 'right'])) {
            throw new Exception('Glb_Query : Join ' . Glb_Text::quote_label('type') . ' attribute should be LEFT, INNER or RIGHT. ' . Glb_Text::quote_label($join['type']) . ' given. '  . print_r($join, true));
        }
        if (empty($join['conditions'])) {
            throw new Exception('Glb_Query : Join ' . Glb_Text::quote_label('conditions') . ' attribute is required. ' . print_r($join, true));
        }
        return $join;
    }

    protected function _normalize_matching($matching, $alias) {
        if (empty($matching['table'])) {
            $matching['table'] = $alias;
        }
        $matching = Glb_Hash::ensure_values($matching, ['type', 'alias', 'conditions'], ['inner', $alias, []]);
        $matching['type'] = strtolower(trim($matching['type']));

        if (!in_array($matching['type'], ['inner', 'left', 'right'])) {
            throw new Exception('Glb_Query : Join ' . Glb_Text::quote_label('type') . ' attribute should be LEFT, INNER or RIGHT. ' . Glb_Text::quote_label($join['type']) . ' given. '  . print_r($join, true));
        }
        /*if (empty($matching['conditions'])) {
            throw new Exception('Glb_Query : Matching ' . Glb_Text::quote_label('conditions') . ' attribute is required. ' . print_r($matching, true));
        }*/
        return $matching;
    }

    /*
     * $return 'this' | 'flat' | 'nested' | 'both'
     */
    public function compile_joins($joins, $query_options = []) {

        $this->_query_options = $query_options;
        $this->_loaded_joins = [];

        if (empty($joins)) { return $this; }
        foreach ($joins as $alias => $item) {
            $item = $this->_normalize_join($item);
            $item['source'] = 'join';
            $item['table_object'] = Glb_Table::get($item['table'], ['alias' => $item['alias']]);
            $this->_loaded_joins[$item['alias']] = $item;
        }

        return $this;
    }

    // resolve fields to be select
    public function compile_fields($required, $fill_default = true) {
        //Glb_Log::info('compile_fields $required ' . $this->alias, $required);

        $this->_required = $required;
        $this->_loaded_fields = [];
        $current_table_results = [];
        $results = [];

        // required can be filled with
        //  - simple : [alias.column_name] or [alias.column_alias => alias.column_name]
        //  - aliased = [alias.column_alias => $formula] or [column_alias => $formula]

        // search for current main table
        //$required = Glb_Array::convert_associative($required, []);
        foreach ($required as $alias => $field) {

            if (is_int($alias)) {

                // then $field must be passed simply formatted alias.column_name or column_name for main table
                $field = Glb_Query::parse_field($field);
                if ($field[1] == '*') { continue; }

                if (empty($field[0]) || $field[0] == $this->sql_alias) {
                    $current_table_results[Glb_Query::alias_field($this->sql_alias, $field[1])] =
                        ['table_alias' => $this->sql_alias, 'column' => $field[1], 'column_alias' => $field[1]];
                    unset($required[$alias]);
                }
                //preg_match("/^[`]*([a-zA-Z0-9\_]*)[`]*[\.]*[`]*([a-zA-Z0-9\_]*)[`]*$/", $field, $matches);
                /*if ($matches[1] == $this->_table['alias'] || ($fill_default && (empty($matches[1])))) {
                    $result[] = ['table_alias' => $this->_table['alias'], 'column' => $matches[2]];
                }*/

            } else {
                $alias_parts = Glb_Array::explode_nudge($alias, '.', 2, '');
                $alias_parts[0] = str_replace('`', '', $alias_parts[0]);
                $alias_parts[1] = str_replace('`', '', $alias_parts[1]);
                if (empty($alias_parts[0]) || $alias_parts[0] == $this->sql_alias) {
                    // search anyway for a simple possibility alias.column_name in $field
                    preg_match("/^[`]*([a-zA-Z0-9\_]*)[`]*[\.]*[`]*([a-zA-Z0-9\_]*)[`]*$/", $field, $matches);
                    $new_result = ['table_alias' => $this->sql_alias, 'column_alias' => $alias_parts[1]];
                    if (empty($matches)) {
                        $new_result['formula'] = $field;
                    } else {
                        $new_result['column'] = $matches[2];
                    }
                    $current_table_results[Glb_Query::alias_field($new_result['table_alias'], $new_result['column_alias'])] = $new_result;
                    unset($required[$alias]);
                }

            }

        }


//        glb_dump($this->_name);
//        glb_dump($this->_db_table);
        if ( (empty($current_table_results) && $fill_default) || empty($this->_required) || in_array('*', $this->_required) || in_array($this->sql_alias . '.*', $this->_required)) {
            if (!empty($this->_db_table['columns'])) {
                foreach ($this->_db_table['columns'] as $column) {
                    $current_table_results[Glb_Query::alias_field($this->sql_alias, $column->name)] = ['table_alias' => $this->sql_alias, 'column' => $column->name, 'column_alias' => $column->name];
                }
            }
        }

        // force primary key
        $primary = $this->primary_key();
        foreach($primary as $primary_column) {
            $column = $this->column($primary_column['column_name']);
            $current_table_results[Glb_Query::alias_field($this->sql_alias, $column->name)] =
                ['table_alias' => $this->sql_alias, 'column' => $column->name, 'column_alias' => $column->name];
        }

        // search all loaded relations fields
        foreach ($this->_loaded_relations as $alias => $relation) {

            //['one_to_one', 'many_to_one']
            if ($relation->is_eager()) {
                // search in
                $sub_required = array_merge($required, $relation->compiled_config('fields', []));
                $results = array_merge($results, $relation->related_table()->compile_fields($sub_required, $fill_default));
            }
        }

        // search all loaded joins fields
        foreach ($this->_loaded_joins as $alias => $join) {
            // search in
            $sub_required = array_merge($required, Glb_Hash::get($join, 'fields', []));
            $results = array_merge($results, $join['table_object']->compile_fields($sub_required, $fill_default));
        }

        // search all loaded joins fields
        /*foreach ($this->_loaded_matchings as $alias => $relation) {
            // search in
            $sub_required = array_merge($required, $relation->compiled_config('fields', []));
            $results = array_merge($results, $relation->related_table()->compile_fields($sub_required, $fill_default));
        }*/

        $this->_required = $required;
        $this->_loaded_fields = $current_table_results;
        //glb_dump('compiled_fields ' . $this->_name . ' ' . print_r($this->_loaded_fields, true));
        $results = array_filter($results);
        return array_merge($current_table_results, $results);

    }

    /**
     * Try to find table definition file, for class Glb_Killer_App_Table for ex.
     *
     * @param string $reference Can be the short name, example "killer_app_objects", or the full name "glb_killer_app_objects"
     * Can also be a raw sql table name, but never add the WP prefix (for multisite support) !
     * The engine will search for tables this way :
     *      1/ Search a file named class-glb-[$reference slugged]-table.php (in caller/core/glb plugins,
     *          except if the plugin was explicitly specified in $table),
     *          then load the Glb_[$table humanized underscored]_Table class that should be defined inside
     *      2/ Search a sql table that is named glb_$reference,
     *          then load the generic Glb_Table class
     *      3/ Search a sql table that is named $reference,
     *          then load the generic Glb_Table class
     *
     * @param string $options
     *      key 'alias' : Will be the sql alias used in sql queries for this table,
     *          and will determine the entity names linked to this table in relations
     *          By default, 'alias' will be the first param $table underscored
     *          This parameter is mostly important for relations builder, because you can build a query pointing on
     *          the same table multiple times, and that need a different sql alias (or mysql will squeak)
     *
     * @return null|Glb_Table object found
     *
     * @usage :
     *      Glb_Table::get('core_users') Will return a Glb_Core_Users_Table object aliased "core_users"
     *      Glb_Table::get('users') Will return a generic Glb_Table object pointing on the mysql "users" table
     *      Glb_Table::get('my_lonely_sql_table_without_class_defined') Will return a generic Glb_Table object
     *      pointing on the mysql "my_lonely_sql_table_without_class_defined" table.
     *      Glb_Table::get('my_nonexistent_sql_table_from_the_moon') returns false, sorry...
     */
    public static function get($reference, $options = []) {

        if ( ($result = self::_resolve('table', $reference)) === false ) {

            return null;

        } else {

            if (!empty($result['plugin'])) {
                $options['plugin'] = $result['plugin'];
            }
            if (!empty($result['reference'])) {
                $reference = $result['reference'];
            }
            if (empty($options['alias'])) {
                $options['alias'] = Glb_Text::underscore($reference);
            }
            $table_object = new $result['class']($reference, $options);
            if (method_exists($table_object, 'initialize')) {
                $table_object->initialize($options);
            }

            // apply late hooks from plugins
            self::apply_hook($table_object, $reference);
            return $table_object;

        }
    }

    public function behaviors() {
        return $this->_behaviors;
    }

    /**
     * Killer function to find an unknown object type in an unknown file, somewhere, somehow
     * @param string $type One of ['behavior', 'table', 'entity']
     * @param $reference
     * @return array|bool
     */
    protected static function _resolve($type, $reference) {

        // initialize
        $plugins = $plugin_aliases = [];

        if (strpos($reference, ':') === false) {
            // if no plugin specified, Search in caller plugin (first), then in core plugin, and then in all Glb plugins
            $plugins[] = Glb_Plugin::get_caller_plugin();
            $plugin_aliases[] = $plugins[0]->alias;

            // core is always registered first
            foreach(Glb_Plugin::get_registered() as $plugin) {
                if (!in_array($plugin->alias, $plugin_aliases)) {
                    $plugins[] = $plugin;
                    $plugin_aliases[] = $plugin->alias;
                }
            }
        } else {
            // if a plugin was explicitly specified
            $splitted = explode(':', $reference, 2);
            $plugins[] = Glb_Plugin::get_registered($splitted[0]);
            $reference = Glb_Text::ensure_leading($splitted[1], $splitted[0] . '_');
        }

        // singularize entity name
        if ($type == 'entity') {
            $reference = Glb_Text::singularize($reference);
        }

        // loop through out plugins
        foreach($plugins as $plugin) {

            // build the class name, only one allowed : Glb_[plugin_alias]_[Item humanized underscored]_[Type]
            // example Glb_Core_Users_Table or Glb_Myplugin_Killerapp_Table

            // add table suffix (_Table or _Entity or _Behavior)
            $class_name = Glb_Text::camelize(Glb_Text::ensure_trailing($reference, '_' . $type), '_');

            // add plugin key
            // $class_name = Glb_Text::ensure_leading($class_name, Glb_Text::camelize($plugin->alias, '_') . '_');

            // add Glb prefix
            $class_name = Glb_Text::ensure_leading($class_name, 'Glb_');

            // if class was already loaded, then return
            if (class_exists($class_name)) {
                return [
                    'plugin' => $plugin,
                    'path' => null,
                    'class' => $class_name,
                    'reference' => $reference,
                ];
            }

            // camelization
            //$class_name = Glb_Text::camelize($class_name, '_');

            // build allowed file paths in /models or models/tables or models/behaviors or models/entities
            //foreach(array_keys($class_names) as $class_name) {
            $file_paths = [];
            $file_name = 'class-' . Glb_Text::slug($class_name) . '.php';

            // search in /models/[types]
            $file_paths[] = $plugin->file_path('models/' . strtolower(Glb_Text::pluralize($type)) . '/' . $file_name);
            // search in /models for lazy plugin developpers
            $file_paths[] = $plugin->file_path('models/' . $file_name);

            // search for classes / files
            foreach ($file_paths as $file_path) {
                if (file_exists($file_path)) {
                    include_once $file_path;
                    if (!class_exists($class_name)) {
                        throw new Exception('Glb_Table: file ' . Glb_Text::quote_label($file_path) .
                            ' was loaded but class '  . Glb_Text::quote_label($class_name) . 'was not found : try again');
                    }
                    return [
                        'plugin' => $plugin,
                        'path' => $file_path,
                        'class' => $class_name,
                        'reference' => $reference,
                    ];
                }
            }
        }

        // if no class was found, search for SQL table
        if ($type == 'table') {
            // if one wants to use a Glb_Table object without declaring a class, one can
            return [
                'plugin' => 'core',
                'path' => null,
                'class' => 'Glb_Table',
                'reference' => $reference,
            ];
        } else if ($type == 'entity') {
            // if one want to use a Glb_Table_Entity object without declaring a class, one can
            return [
                'plugin' => 'core',
                'path' => null,
                'class' => 'Glb_Table_Entity',
                'reference' => $reference,
            ];
        }

        return false;

    }

    public function add_behavior($behavior, $options = []) {

        //var_dump(Glb_Array::simplify(debug_backtrace(), ['file', 'line', 'function']));
        //var_dump($behavior);
        //var_dump('searching for $behavior ' . $behavior . ' ' . json_encode($options));

        $result = Glb_Cache::get('Glb_Table:add_behavior:' . $behavior . ':' . json_encode($options));

        if ($result !== null) { return $result; }

        if ( ($behavior_result = self::_resolve('behavior', $behavior)) === false ) {
            //var_dump(' !!! can not find it ' . $behavior);
            return false;
        } else {
            //var_dump(' !!! loaded ' . $behavior);
            //Glb_Cache::set('Glb_Table:get:' . $table . ':' . json_encode($options), 'ok');
            $behavior_object = new $behavior_result['class']($behavior, $options);
            Glb_Cache::set('Glb_Table:add_behavior:' . $behavior . ':' . json_encode($options), $behavior_object);
            $behavior_object->initialize($options);
            $this->_behaviors[$behavior] = $behavior_object;
            return Glb_Cache::set('Glb_Table:add_behavior:' . $behavior . ':' . json_encode($options), $behavior_object);

            //var_dump(' !!! loaded2 ' . $table);
            //var_dump(' !!! loaded2 ' . Glb_Cache::get('Glb_Table:get:' . $table . ':' . json_encode($options)));

        }
        //var_dump(' !!! loaded result ' . print_r(Glb_Cache::get('Glb_Table:get:' . $table . ':' . print_r($options, true)), true));
        //return Glb_Cache::get('Glb_Table:get:' . $table . ':' . print_r($options, true));
    }

    public function remove_behavior($behavior) {
        unset($this->_behaviors[$behavior]);
    }

    public function joins() {
        return $this->_loaded_joins;
    }

    /*public function matchings() {
        return $this->_loaded_matchings;
    }*/

    public function one_to_many($alias, $config = []) {
        $relation = new Glb_Relation($this, $alias, __FUNCTION__, $config);
        $this->_relations[$alias] = $relation;
        return $this;
    }

    public function one_to_one($alias, $config = []) {
        $relation = new Glb_Relation($this, $alias, __FUNCTION__, $config);
        $this->_relations[$alias] = $relation;
        return $this;
    }

    public function many_to_many($alias, $config = []) {
        $relation = new Glb_Relation($this, $alias, __FUNCTION__, $config);
        $this->_relations[$alias] = $relation;
        return $this;
    }

    public function many_to_one($alias, $config = []) {
        $relation = new Glb_Relation($this, $alias, __FUNCTION__, $config);
        $this->_relations[$alias] = $relation;
        return $this;
    }

    /*protected function _normalize_relation_options($alias, $type, $options) {

        Glb_Log::info('_normalize_relation_options : '  . $alias);

        $default_options = [
            'type' => $type,
            'binding_key' => null,
            'foreign_key' => null,
            'property' => null,
            'table' => null,
            'entity' => null,
            'alias' => $alias,
        ];

        if (!empty($options['table'])) {
            $options['table_name'] = $options['table'];
        }

        $options = array_merge($default_options, $options);

        if (empty($options['table_name'])) {
            // @todo : try remove/add plugin name
            $options['table_name'] = Glb_Text::camelize($alias);
        } else {

        }

       //$options['table'] = Glb_Table::get($options['table']);

        if (empty($options['binding_key'])) {
            if (in_array($type, ['many_to_one', 'one_to_many'])) {
                Glb_Log::info('_normalize_relation_optionsx : '  . $alias);
                Glb_Log::info('_normalize_relation_optionsy : '  . Glb_Text::singularize(strtolower($alias)));
                // @todo : try remove plugin name
                $options['binding_key'] = Glb_Text::singularize(strtolower($alias)) . '_id';
            } else if (in_array($type, ['one_to_one', 'one_to_many'])) {
                $options['binding_key'] = 'id';
            }
        }

        if (empty($options['foreign_key'])) {
            if (in_array($type, ['many_to_one'])) {
                $options['foreign_key'] = 'id';
            } else if (in_array($type, ['one_to_one', 'one_to_many'])) {
                $options['foreign_key'] = Glb_Text::singularize(strtolower($alias)) . '_id';
            }
        }

        if (empty($options['property'])) {
            if (in_array($type, ['many_to_one'])) {
                $options['property'] = Glb_Text::singularize(strtolower($alias));
            } else if (in_array($type, ['many_to_many', 'one_to_many'])) {
                $options['property'] = Glb_Text::pluralize(strtolower($alias));
            }
        }

        if (empty($options['entity'])) {
            $options['entity'] = Glb_Text::singularize(strtolower($alias));
        }

        Glb_Log::info('_normalize_relation_options3 : ', $options);

        return $options;

    }*/

    /*protected function _calc_relation_key($alias, $value) {
        return (empty($value) ? Glb_Text::singularize(strtolower($alias)) . '_id' : $value);
    }
    protected function _calc_id_key($alias, $value) {
        return (empty($value) ? Glb_Text::singularize(strtolower($alias)) . '_id' : $value);
    }*/

    public function query($type, $options = []) {
        return new Glb_Query($type, $this, $options);
    }

    /**
     * Shortcut for Glb_Table::query('select')
     * @param array $options
     * @return Glb_Query
     */
    public function select($options = []) {
        return new Glb_Query('select', $this, $options);
    }

    public function columns() {
        return $this->_db_table['columns'];
    }

    public function column($column) {
        if (empty($column)) {
            return null;
        }
        if ($this->has_column($column)) {
            return $this->_db_table['columns'][$column];
        } else {
            return null;
        }
    }

    /*public function __get($property) {
        var_dump('__get ' . $property);
        if ($property == 'table') {
            $property = 'name';
        }
        if (in_array($property, ['alias', 'name', 'object'])) {
            return $this->_table[$property];
        } else {
            return false;
        }
    }

    public function __set($property, $value) {
        var_dump('__set ' . $property . '=' . $value);
        if ($property == 'table') {
            $property = 'name';
        }
        if (in_array($property, ['alias', 'name', 'object'])) {
            $this->_table[$property] = $value;
            return $value;
        } else {
            return false;
        }
    }*/

    public function has_column($column) {
        return !empty($this->_db_table['columns'][$column]);
    }

    public function before_select($query, $context, $options) {

    }

    public function after_select($query, $context, $options) {

    }

    public function before_update($entity, $context, $options) {

    }

    public function after_update($entity, $context, $options) {

    }

    public function before_insert($entity, $context, $options) {

    }

    public function after_insert($entity, $context, $options) {

    }

    public function before_delete($entity, $context, $options) {

    }

    public function after_delete($entity, $context, $options) {

    }

    /*public function __toString() {
        return 'ok';
    }

    public function __debugInfo() {
        return [
            //'xok' => 'xok',
        ];
    }*/

    /**
     * Simplify debug
     */
    public function __debugInfo() {
        return [
            'reference' => $this->reference,
            'name' => $this->name,
            'alias' => $this->alias,
            'sql_name' => $this->sql_name,
            'sql_alias' => $this->sql_alias,
            'entity' => $this->entity,
            'behaviors' => $this->_behaviors,
            'loaded_relations' => $this->_loaded_relations,
            'loaded_joins' => $this->_loaded_joins,
            //'loaded_matchings' => $this->_loaded_matchings,
            'loaded_fields' => $this->_loaded_fields
        ];
    }

    public static function hook($reference, $method, $alias, $config) {
        if (strpos($reference, ':') !== false) {
            $reference = explode(':', $reference, 2)[1];
        }
        Glb_Hash::ensure_values(self::$_hooks, $reference, []);
        self::$_hooks[$reference][] = [
            'method' => $method,
            'alias' => $alias,
            'config' => $config,
        ];
    }


    public static function apply_hook($table_object, $reference) {
        if (strpos($reference, ':') !== false) {
            $reference = explode(':', $reference, 2)[1];
        }
        if (!empty(self::$_hooks[$reference])) {
            foreach(self::$_hooks[$reference] as $hook) {
                $table_object->{$hook['method']}($hook['alias'], $hook['config']);
            }
        }
    }

}