<?php

/*
 * Glb_Table class.
 */

use Cake\Datasource\EntityInterface;

class Glb_Table_Entity extends Glb_Entity implements JsonSerializable
{
    protected $_table                = null;
    protected $_new                  = false;
    protected $_skip_validation      = true;

    /**
     * List of property names that should **not** be computed in JSON or Array
     * representations of this Entity.
     * Properties are filtered when jsonSerialize(), toArray() or functions are called
     *
     * @var array
     */
    protected $_hidden         = [];

    /**
     * List of virtual fields that **should** be included in JSON or array
     * representations of this Glb_Entity. If a field is present in both _hidden and _virtual
     * the field will **not** be in the array/json representations of the entity.
     * Properties are filtered when jsonSerialize(), toArray() or __toString() functions are called
     *
     * @var array
     */
    protected $_virtual        = [];
    protected $_errors         = [];
    protected $_warnings       = [];

    public function __construct($table, $data, $is_new, $options = [])
    {

        $options = Glb_Array::convert_associative($options);
        $this->_table = $table;
        $this->_new = $is_new;
        Glb_Validator::check_value($data, 'empty', []);
        $this->_skip_validation = (!empty($options['skip_validation']));
        parent::__construct($data);
        //$this->patch($data);
        $this->_validate_values();
        $this->_skip_validation = false;

    }

    public function __debugInfo() {
        $info = parent::__debugInfo();
        $extra = [];
        if (!empty($this->_errors)) {
            $extra += ['~has_error' => (!empty($this->_errors)), '~errors' => $this->_errors];
        } else {
            $extra += ['~has_error' => false];
        }
        if (!empty($this->_warnings)) {
            $extra += ['~has_warning' => (!empty($this->_warnings)), '~warnings' => $this->_warnings];
        } else {
            $extra += ['~has_warning' => false];
        }
        return $info + [ '~is_new' => !empty($this->_new) ] + $extra;
    }

    public function is_new()
    {
        return $this->_new;
    }

    public function method_exists($property)
    {
        if (method_exists($this, $property)) {
            return true;
        }
        return false;
    }

    /*public function virtual($method, $params = [], $default = null) {
        if (method_exists($this, '__' . $method)) {
            return call_user_func_array([$this, '__' . $method], $params);
        }
        return $default;
    }*/

    public function __($method, $params = [], $default = null)
    {
        if (method_exists($this, '__' . $method)) {
            return call_user_func_array([$this, '__' . $method], $params);
        }
        return $default;
    }

    public function first_access($type, $property = null, $value = null)
    {
        parent::first_access($type, $property, $value);
        $this->_table->complete_late($this);
    }

    protected function _validate_values()
    {
        if ($this->_skip_validation || empty($this->_properties)) {
            return true;
        }
        foreach($this->_table->columns() as $column) {
            if ($this->exists($column->name)) {
                $this->_validate_value($column->name, $this->_get($column->name), $column);
            }
        }
    }

    protected function _validate_value($property, $value, $column = null) {
        if ($this->_skip_validation) {
            return true;
        }
        if ($column === null) {
            $column = $this->_table->column($property);
        }

        // no column found, do what you want...
        if ($column === null) {
            return;
        }

        //glb_dump('validate ' . print_r($column, true));
        if ( !$column->validate($value) ) {
            if ($column->has_warning()) {
                Glb_Hash::ensure_values($this->_warnings, $property, []);
                $this->_warnings[$property] = array_merge($this->_warnings[$property], $column->warnings());
            }
            if ($column->has_error()) {
                Glb_Hash::ensure_values($this->_errors, $property, []);
                $this->_errors[$property] = array_merge($this->_errors[$property], $column->errors());
            }
            return false;
        }
        return true;
    }

    public function has_error() {
        return !empty($this->errors);
    }

    public function errors() {
        return $this->_errors;
    }

    public function has_warning() {
        return !empty($this->_warnings);
    }

    public function warnings() {
        return $this->_warnings;
    }

    public function __set($property, $value) {
        parent::__set($property, $value);
        $this->_validate_value($property, $value);
    }

    protected function &_get($property)
    {
        $this->_check_access('get', $property);

        //glb_dump($this->_properties);
        if (!parent::exists($property) && !$this->method_exists('__' . $property)) {
            // search for relations in parent table
            throw new Exception('Glb_Table_Entity unable to find property ' . $property);
        }

        if ($this->method_exists('__' . $property)) {
            $this->_properties[$property] = $this->__($property);
        }
        return $this->_properties[$property];
    }

    /*public function __debugInfo() {
        $this->_check_access('get');
        return $this->_properties + [
                '~is_modified' => !empty($this->_modified),
                '~is_new' => $this->_new,
                '~modified' => $this->_modified,
            ];
    }*/

    public function jsonSerialize()
    {
        return json_encode($this->extract($this->visible_properties(), true), JSON_PRETTY_PRINT);
    }

    public function to_array()
    {
        return $this->extract($this->visible_properties(), true);
    }

    /**
     * Returns a string representation of this object in a human readable format.
     *
     * @return string
     */
    public function __toString()
    {
        return json_encode($this->extract($this->visible_properties(), true), JSON_PRETTY_PRINT);
    }

    public function visible_properties()
    {
        $properties = array_keys($this->_properties);
        $properties = array_merge($properties, $this->_virtual);
        return array_diff($properties, $this->_hidden);
    }

    public function properties()
    {
        return array_keys($this->_properties);
    }

    public function extract($properties, $recursive = true)
    {
        $result = [];
        foreach ($properties as $property) {
            $value = $this->_properties[$property];
            if ($recursive && is_object($value) && (in_array(get_class($value), ['Glb_Table_Entity', 'Glb_Table_Entity']))) {
                $result[$property] = $value->extract($value->visible_properties(), true);
            } elseif ($recursive && is_object($value) && (in_array(get_class($value), ['Glb_Db_Collection', 'Glb_Collection']))) {
                $result[$property] = [];
                foreach ($value->to_array() as $item_key => $item_value) {
                    if (get_class($item_value) == 'Glb_Table_Entity' || get_parent_class($item_value) == 'Glb_Table_Entity') {
                        $result[$property][$item_key] = $item_value->extract($item_value->visible_properties(), true);
                    } else {
                        $result[$property][$item_key] = $item_value;
                    }
                }
            } else {
                $result[$property] = $value;
            }
        }
        return $result;
    }

    public function skip_validation($value = null) {
        if ($value === null) {
            return $this->_skip_validation;
        }
        $this->_skip_validation = $value;
        return $this;
    }

    public function patch($data, $options = []) {
        if (empty($data)) {
            return $this;
        }
        $options = Glb_Array::convert_associative($options);
        $this->skip_validation(!empty($options['skip_validation']));
        foreach($this->_table->columns() as $column) {
            if (array_key_exists($column->name, $data)) {
                $this[$column->name] = $data[$column->name];
            }
        }
        $this->skip_validation(false);
        return $this;
    }

    public function reset_warnings() {
        $this->_warnings = [];
        return $this;
    }
    public function reset_errors() {
        $this->_errors = [];
        return $this;
    }

    public function set_errors($errors) {
        $this->_errors = $errors;
        return $this;
    }

    public function set_warnings($warnings) {
        $this->_warnings = $warnings;
        return $this;
    }

    public function set_new($new) {
        $this->_new = $new;
        return $this;
    }

    public function reset_new() {
        $this->_new = false;
        return $this;
    }

}
