<?php

class Glb_Query
{

    private $_table             = null;
    private $_type              = null;

    protected $_options         = [];
    private $_executed          = false;
    private $_data              = [];
    private $__cache            = [];
    private $_stopped           = 0;    // 1 : stop dispatch, 2: stop query

    protected $_templates = [
        'queries' => [
            'select' => "{select}{modifiers}{fields}{from}{join}{where}{group}{having}{order}{limit}{offset}{epilog}",
            'delete' => "{delete}{modifiers}{from}{where}{order}{limit}{epilog}",
            'update' => "{update}{modifiers}{table}{set}{join}{where}{order}{limit}{epilog}",
            'insert' => "{insert}{modifiers}{into}{fields}{values}{epilog}",
            // Optimize & truncate. Sometimes, it can be good (but only sometimes !)
            'optimize' => '{optimize}{modifiers}{table}{epilog}',
            'truncate' => '{truncate}{table}{epilog}',

        ],
        'statements' => [
            'select' => 'SELECT',
            'delete' => 'DELETE',
            'optimize' => 'OPTIMIZE',
            'truncate' => 'TRUNCATE',
            'update' => 'UPDATE',
            'insert' => 'INSERT',

            'fields' => ' %s',
            'modifiers' => ' %s',
            'table' => ' %s',
            'from' => "\n FROM %s",
            'join' => "\n %s",
            'where' => "\n WHERE %s",
            'group' => "\n GROUP BY %s",
            'having' => "\n HAVING %s",
            'order' => "\n ORDER BY %s",
            'limit' => "\n LIMIT %s",
            'offset' => "\n OFFSET %s",
            'epilog' => "\n %s",
            'set' => "\n SET %s",
            'into' => "\n INTO %s",
            'values' => "\n VALUES %s",
        ]
    ];

    function __construct($type, $parent_table, $options = []) {
        $this->_type = strtolower($type);
        $this->_table = $parent_table;
        $this->_options = array_merge($this->_options, $options);
        $this->_reset_all();
    }

    //private $_debug = false;
    private $_debug = false;
    private function _debug(...$messages) {

        if (empty($this->_debug)) {
            return;
        }
        $arg_count = func_num_args();
        $arg_list = func_get_args();
        $message = '';
        for ($i = 0; $i < $arg_count; $i++) {
            $message .= ' ' . print_r($arg_list[$i], true);
        }

        if ($this->_debug == 'dump') {
            glb_dump($message);
        } else {
            Glb_Log::info($message);
        }
    }

    /*
     * strategy
     *      - merge : add to actual array
     *      - overwrite : replace current values
     */
    private function _set($property, $value, $overwrite = false, $no_reset = false) {

        if (!$no_reset) {
            $this->_reset_status();
            $this->_reset_cache();
        }
        if ($value === null) { $this->_data[$property] = []; return $this; }
        if ($value === []) { return $this; }

        // force scalar values to be arrays
        Glb_Array::ensure($value);

        // force key with empty array
        Glb_Array::ensure_key($this->_data, $property, []);

        if ($overwrite) {
            $this->_data[$property] = $value;
        } else {
            // check if this value doesn't already exists, associated with a numerical key
            $this->_data[$property] = array_merge($this->_data[$property], $value);
        }
        return $this;

    }

    /*
     * strategy
     *      - array : returns the array
     *      - last : return the last value of the array
     *      - first : return the first value of the array
     */
    private function _get($property, $default = null, $strategy = "array") {

        if (!array_key_exists($property, $this->_data)) {
            return $default;
        } else {
            if ($strategy == 'array') {
                return $this->_data[$property];
            } else {
                if (empty($this->_data[$property])) {
                    return $default;
                } else {
                    if ($strategy == 'last') {
                        return $this->_data[$property][count($this->_data[$property]) - 1];
                    } else if ($strategy == 'first') {
                        return $this->_data[$property][0];
                    }
                }
            }
        }

    }

    public function where($values, $overwrite = false) {
        return $this->_set('where', $values, $overwrite);
    }

    public function order($values, $overwrite = false) {
        return $this->_set('order', $values, $overwrite);
    }

    public function group($values, $overwrite = false) {
        return $this->_set('group', $values, $overwrite);
    }

    public function set($values, $overwrite = false) {
        return $this->_set('set', $values, $overwrite);
    }

    public function having($values, $overwrite = false) {
        return $this->_set('having', $values, $overwrite);
    }

    public function limit($value, $overwrite = true) {
        return $this->_set('limit', $value, $overwrite);
    }

    public function offset($value, $overwrite = true) {
        return $this->_set('offset', $value, $overwrite);
    }

    /**
     * Matching is used for joining eager/inner a relation that should normally be built later
     * @param $values
     * @param bool $overwrite
     * @return Glb_Query
     */
    public function matching($value, $overwrite = false) {
        if (is_string($value)) { $value = [$value]; }
        $value = Glb_Array::convert_associative($value, []);
        $value = Glb_Array::ensure_keys($value, 'source', 'matching');
        $value = Glb_Array::ensure_keys($value, 'strategy', 'eager');
        return $this->_set('relations', $value, $overwrite);
    }

    public function not_matching($value, $overwrite = false) {
        if (is_string($value)) { $value = [$value]; }
        $value = Glb_Array::convert_associative($value, []);
        $value = Glb_Array::ensure_keys($value, 'source', 'not_matching');
        $value = Glb_Array::ensure_keys($value, 'strategy', 'eager');
        return $this->_set('relations', $value, $overwrite);
    }

    public function fields($values, $overwrite = false) {
        return $this->_set('fields', $values, $overwrite);
    }

    public function values($values, $overwrite = false) {
        return $this->_set('values', $values, $overwrite);
    }

    public function modifiers($values, $overwrite = false) {
        return $this->_set('modifiers', $values, $overwrite);
    }

    public function into($value, $overwrite = true) {
        return $this->_set('into', $value, $overwrite);
    }

    public function epilog($value, $overwrite = true) {
        return $this->_set('epilog', $value, $overwrite);
    }

    /**
     * @param $values
     * @param bool $overwrite
     * @return Glb_Query
     * @usage
     *      $query->join([
     *           'table' => 'authors_infos',
     *           'alias' => 'author_info',
     *           'type' => 'inner',
     *           'conditions' => '(`author_info`.`author_id` = `author`.`id`)'
     *      ]);
     */
    public function join($value, $overwrite = false) {
        if (!is_array($value)) {
            throw new Exception('Glb_Query : Join item must be an array, no choice.');
        }
        if (array_key_exists('table', $value)) {
            $value = [$value];
        }
        $value = Glb_Array::ensure_keys($value, 'source', 'join');
        $value = Glb_Array::ensure_keys($value, 'strategy', 'eager');

        $this->_set('join', $value, $overwrite);
        return $this;
    }

    // extract fields with tables as 0 and field name as 1
    /* protected function _extract_fields($fields) {
         $result = [];
         foreach($fields as $field) {
             if (strpos($field, '.') === false) {
                 $field = [$this->_table->get_name(), $field];
             } else {
                 $field = explode('.', $field, 2);
             }
             if (empty($result[$field[0]])) {
                 $result[$field[0]] = [];
             }
             $result[$field[0]][] = $field;
         }
         return $result;
     }*/

    /*
     * [
     *      'fields' => [...],
     *      ''
     * ]
     *
     * @usage :
     *      Glb_Table::get('mytable')->query('select')->contain('mytable2');
     *      Glb_Table::get('mytable')->query('select')->contain(['mytable2', 'mytable3']);
     *      Glb_Table::get('mytable')->query('select')->contain(
     *              ['mytable2' => [
     *                  'conditions' => ['Mytable2.field' => 'Mytable.field'],
     *                  'fields' => ['Mytable2.field1', 'Mytable2.field2', 'Mytable2.field3'],
     *                  'type' => 'left'
     *              ],
     *              'mytable3']);
     *
     */
    // @todo : alias can be composed
    /**
     * @param array $alias
     * @param array $options
     * @return $this
     * @usage
     *  Glb_Table::get('my_table')->query('select')->contain(
     *      [
     *          'my_relation1',
     *          'my_relation2' => [
     *              // all attributes will be applied to query attributes calling the function with the same name
     *              // only the "conditions" attribute will apply to table relation (added to the sql JOIN conditions)
     *              'fields' => ['my_table_join2.id'],
     *              'where' => ['my_table_join2.deleted IS' => null],
     *              'limit' => [5, 10],
     *              'conditions' => ['my_table_join2.id >' => 3],
     *          ],
     *          'my_relation3.my_relation4' => [],
     *          'my_relation2_bis' => [
     *              'relation' => 'my_relation2',
     *          ]
     *      ]
     * )
     */
    public function contain($contain, $overwrite = false) {
        if (empty($contain)) { return $this; }
        if (is_string($contain)) { $contain = [$contain]; }
        $contain = Glb_Array::convert_associative($contain, []);

        // reformat contain in a fully nested array
        $compiled = [];
        foreach($contain as $contain_key => $contain_value) {
            $new_value = [];
            if (!empty($contain_value)) {
                foreach($contain_value as $contain_value_key => $contain_value_value) {
                    Glb_Hash::set($new_value, $contain_value_key, $contain_value_value);
                }
            }
            Glb_Hash::ensure_values($new_value, 'source', 'contain');
            $contain_key = str_replace('.', '.contain.', $contain_key);
            Glb_Hash::set($compiled, $contain_key, $new_value);
        }
        $this->_set('relations', $compiled, $overwrite);
        return $this;
    }

    /**
     * ['ignoreCallbacks.tables' => ['test5']
     * @param $options
     * @return $this
     *
     * @usage
     *      $query->options(['ignore_callbacks' => 1]);
     *      OR
     *      $query->options([
     *          'ignore_callbacks.tables' => ['documents', 'users_documents']
     *          'ignore_callbacks.behaviors' => ['who_when']
     *      ]);
     */
    public function options($options) {
        $this->_reset_status();
        foreach($options as $option_key => $option_value) {
            Glb_Hash::set($this->_options, $option_key, $option_value);
        }
        return $this;
    }

    private function _cache($key, $value = null) {
        if ($value === null) {
            return Glb_Hash::get($this->__cache, $key);
        } else {
            return Glb_Hash::set($this->__cache, $key, $value);
        }
    }


    /**
     *  ->join([
     *      'table' => 'users',
     *      'alias' => 'users_alias',
     *      'type' => 'LEFT',
     *      'fields' => ['ID', 'user_login', 'user_email'],
     *      'conditions' => 'users_alias.ID = main_table.user_id',
    ])
     */

    protected function _reset_all() {
        $this->_reset_status();
        $this->_reset_cache();
        $this->_reset_data();
    }

    protected function _reset_status() {
        $this->_executed = false;
        $this->_stopped = 0;
    }

    protected function _compile_init() {

        // reset buffers
        $this->_reset_status();
        $this->_reset_cache();

        //Glb_Log::info('before dedupe all ', $this->_data);
        // dedupe data
        foreach($this->_data as $data_key => $data_value) {
            if (!in_array($data_key, ['fields', 'values'])) {
                $this->_data[$data_key] = Glb_Array::dedupe($this->_data[$data_key]);
            }
        }
        //Glb_Log::info('after dedupe all ', $this->_data);
    }

    protected function _reset_cache() {
        $this->_tmp = [];
    }

    protected function _reset_data() {
        $this->_data = [];
    }

    public function execute() {

        if ($this->_executed) {
            return $this->_cache('results');
        }

        $this->_build_sql();
        if ($this->is_stopped()) { return null; }

        // Run sql
        $rows = Glb_Db::instance()->raw_query($this->_cache('sql'), []);

        // Build entities
        if ($this->_type == 'select') {
            $results = $this->_table->build_entities($rows, false);
        } else if ($this->_type == 'insert') {
            $results = Glb_Db::instance()->last_insert_id();
        } else if (in_array($this->_type, ['update', 'delete'])) {
            $results = Glb_Db::instance()->rows_affected();
            glb_dump($results);
        }

        // Log results
        $this->_cache('results', $results);

        // Dispatch after callbacks
        $this->_dispatch('after');
        if ($this->is_stopped()) { return null; }

        $this->_executed = true;

        // return
        return $this->_cache('results');

    }

    public function last_warnings() {
        return Glb_Db::instance()->last_warnings();
    }

    public function last_errors() {
        return Glb_Db::instance()->last_errors();
    }

    protected function _dispatch_one($key, $hook, $event) {

        call_user_func($hook, $this, $event, $this->_options);

        if ($event->is_query_stopped()) {

            $this->_reset_all();
            $this->_stopped = 2;
            throw new Exception('Query stopped from ' . $key . ' : ' . $hook[1]);

        } else if ($event->is_dispatch_stopped()) {

            $this->_stopped = 1;
            throw new Exception('Dispatch stopped from ' . $key . ' : ' . $hook[1]);

        } else {

            return true;

        }
    }

    /**
     * @todo : manage insert / delete / update
     * @param $method
     * @param $options
     * @return bool
     */
    protected function _dispatch($method) {

        try {
            $method = $method . '_' . $this->_type;
            if (in_array(Glb_Hash::get($this->_options, 'ignore_callbacks', false), [true, 1], true)) {
                return true;
            }

            $event = new Glb_Query_Event('query', $method, ['primary' => true, 'primary_table' => $this->_table, 'related_table' => null, 'relation' => null]);

            // find current table event
            if (!in_array($this->_table->reference, Glb_Hash::get($this->_options, 'ignore_callbacks.tables', []))) {

                if (method_exists($this->_table, $method)) {
                    $this->_dispatch_one('table.' . $this->_table->reference, [$this->_table, $method], $event);
                }

                // find main table behavior's events
                foreach ($this->_table->behaviors() as $behavior_key => $behavior) {
                    if (!in_array($behavior_key, Glb_Hash::get($this->_options, 'ignore_callbacks.behaviors', []))) {
                        if (method_exists($behavior, $method)) {
                            $this->_dispatch_one('behavior.' . $this->_table->reference, [$this->_table, $method], $event);
                        }
                    }
                }
            }

            $event->data('primary', false);

            // find joined table event first
            if (!empty($this->_cache('relations'))) {
                foreach ($this->_cache('relations') as $relation) {

                    if (!$relation->is_eager() || in_array($relation->alias, Glb_Hash::get($this->_options, 'ignore_callbacks.tables', []))) {
                        continue;
                    }

                    $event->data('related_table', $relation->related_table())->data('relation', $relation);

                    if (method_exists($relation->related_table(), $method)) {
                        $this->_dispatch_one('table.' . $relation->related_table()->reference, [$relation->related_table(), $method], $event);
                    }

                    // find related table behavior's events
                    foreach ($relation->related_table()->behaviors() as $behavior_key => $behavior) {
                        if (!in_array($behavior_key, Glb_Hash::get($this->_options, 'ignore_callbacks.behaviors', []))) {
                            if (method_exists($behavior, $method)) {
                                $this->_dispatch_one('behavior.' . $behavior_key, [$behavior, $method], $event);
                            }
                        }

                    }

                }
            }

        } catch (Exception $ex) {
            Glb_Log::error('error dispatching ', $ex);
        }

    }

    public function is_stopped() {

        return ($this->_stopped == 2);

    }

    private function _build_sql() {

        // Pre-compile for the callbacks
        $this->_cache('sql', $this->_compile());

        // Dispatch before callbacks
        $this->_dispatch('before');
        if ($this->is_stopped()) { return null; }

        // Re-Compile for the query taking into account the callbacks
        return $this->_cache('sql', $this->_compile());

    }

    public function sql() {

        return $this->_build_sql();

    }

    protected function _build_conditions($conditions, $operator = 'AND', $level = 0) {
        //var_dump('$conditions ' . print_r($conditions, true));
        //var_dump('%s ' . print_r(esc_sql('%s '), true));

        if (empty($conditions) || !is_array($conditions)) {
            return '';
        }

        //Glb_Log::info('build_conditions ' . $level, $operator, $conditions);
        $operators = ['>', '>=', '<', '<=', '<>', '!=', '<=>', '=', 'BETWEEN', 'IN', 'IS', 'NOT', 'LIKE', 'REGEXP', 'RLIKE', 'REGEXP_LIKE'];

        $condition_items = [];

        foreach($conditions as $condition_key => $condition_value) {

            //if (is_string($condition_key)) { $condition_key = trim($condition_key); }
            //if (is_string($condition_value)) { $condition_value = trim($condition_value); }

            //Glb_Log::info('testing ' . $operator . ' ' . $level . ' ' . print_r($condition_key, true) . ' : ' . print_r($condition_value, true));

            // if a logical operator is found, recurse that
            //Glb_Log::info('$condition_key ' . $condition_key . ' ' . $level  . ' ' . $operator . ' ' . print_r($condition_value, true));
            if (in_array(strtoupper($condition_key), ['XOR', 'OR', 'AND', '||', '&&'])) {

                $condition_items[] = $this->_build_conditions($condition_value, strtoupper($condition_key), $level + 1);

            } else if (!is_int($condition_key)) {
                // may be a string ?

                // check if there is a valid operator at the end
                $splitted = explode(' ', trim(strtolower($condition_key)));
                if (!in_array(strtoupper($splitted[count($splitted) - 1]), $operators)) {
                    $splitted = [$condition_key];
                } else {
                    $splitted[count($splitted) - 1] = strtoupper($splitted[count($splitted) - 1]);
                }

                if (count($splitted) == 1) {
                    $condition_items[] = trim($condition_key) . ' = ' . $this->_build_conditions_value('', $condition_value);
                } else {
                    $condition_items[] = implode(' ', $splitted) . ' ' . $this->_build_conditions_value($splitted[count($splitted) - 1], $condition_value);
                }

            } else {
                //Glb_Log::info('is int ');

                if (is_array($condition_value)) {
                    $condition_items[] = $this->_build_conditions($condition_value, 'AND', $level + 1);
                } else {
                    $condition_items[] = $condition_value;
                }

            }

        }

        //Glb_Log::info('return _build_conditions ' . $level, $operator, $condition_items);
        if ($level == 0) {
            return implode(' ' . $operator . ' ', $condition_items);
        } else {
            return '(' . implode(' ' . $operator . ' ', $condition_items) . ')';
        }

    }

    /**
     * Check if a relation is matching for renaming
     * @param $string
     */
    protected function _check_matching($string) {
        $relations = $this->_get('relations', []);
        $relation_keys = array_keys($relations);

        foreach($relations as $relation_key => $relation_value) {
            //$relation = $this->_table->relation
            if ($relations->source == 'matching') {
            }
        }

        glb_dump($relations);
        /*foreach($relations as $relation) {

        }

        $original_alias = Glb_Text::remove_leading($this->alias, '_matching_');
        $filtered2 = [];
        foreach($this->_config['final'][$config_key] as $condition_key => $condition_value) {
            if (is_string($condition_key)) {
                $condition_key = preg_replace('/\b' . $original_alias . '\b/', $this->alias, $condition_key);
                //glb_dump($condition_key);
            }
            if (is_string($condition_value)) {
                $condition_value = preg_replace('/\b' . $original_alias . '\b/', $this->alias, $condition_value);
                //glb_dump($condition_value);
            }
            $filtered2[$condition_key] = $condition_value;
        }*/

    }

    protected function _variable_type($variable) {
        if (is_int($variable)) {
            return '%d';
        } else if (is_float($variable)) {
            return '%f';
        } else {
            return '%s';
        }
    }

    protected function _build_insert_values($values, $fields) {
        $values_nested_level = Glb_Array::nested_level($values);
        $result = [];
        if ($values_nested_level == 1) {
            // only one row
            $result_row = [];
            foreach($values as $value) {
                if ($value === null) {
                    $result_row[] = 'NULL';
                } else {
                    $result_row[] = Glb_Db::instance()->prepare($this->_variable_type($value), [$value]);
                }
            }
            $result[] = implode(', ', $result_row);
        } else if ($values_nested_level == 2) {
            // multiple rows
            foreach($values as $row_values) {
                $result_row = [];
                foreach($row_values as $value) {
                    if ($value === null) {
                        $result_row[] = 'NULL';
                    } else {
                        $result_row[] = Glb_Db::instance()->prepare($this->_variable_type($value), [$value]);
                    }
                    //$result_row[] = Glb_Db::instance()->prepare($this->_variable_type($value), [$value]);
                }
                $result[] = implode(', ', $result_row);
            }
        }
        return $result;

    }

    protected function _build_update_values($values, $fields) {
        // only one row
        $result_row = [];
        foreach($fields as $index => $field) {
            if ($values[$index] === null) {
                $result_row[] = self::quote_field($field) . ' = NULL';
            } else {
                $result_row[] = self::quote_field($field) . ' = ' . Glb_Db::instance()->prepare($this->_variable_type($values[$index]), [$values[$index]]);
            }
        }
        $result = implode(', ', $result_row);
        return $result;
    }
    //The following placeholders can be used in the query string: %d (integer) %f (float) %s (string)

    protected function _build_conditions_value($operator, $condition_value) {


        if (is_null($condition_value)) {
            return 'NULL';
        }

        if (in_array($operator, ['LIKE', 'RLIKE', 'REGEXP', 'REGEXP_LIKE']) || is_string($condition_value)) {
            return Glb_Db::instance()->prepare($this->_variable_type($condition_value), $condition_value);
        }

        if (in_array($operator, ['BETWEEN']) && is_array($condition_value)) {
            return Glb_Db::instance()->prepare($this->_variable_type($condition_value[0]) . ' AND ' . $this->_variable_type($condition_value[1]), $condition_value);
        }

        if (in_array($operator, ['IN']) && is_array($condition_value)) {
            $condition_types = array_map(function($val) { return $this->_variable_type($val); }, $condition_value);
            return '(' . Glb_Db::instance()->prepare(implode(', ', $condition_types), $condition_value) . ')';
        }

        if (is_bool($condition_value)) {
            return ($condition_value ? 'true' : 'false');
        }

        return Glb_Db::instance()->prepare($this->_variable_type($condition_value), $condition_value);

    }




    /*protected function _field_alias($table_alias, $column) {
        return $table_alias . $this->_options['alias_delimiter'] . $column;
    }

    protected function _field_unalias($field_alias) {
        return explode($this->_options['alias_delimiter'], $field_alias, 2);
    }*/

    /**
     *  ->join([
     *      'table' => 'users',
     *      'alias' => 'users_alias',
     *      'type' => 'LEFT',
     *      'fields' => ['ID', 'user_login', 'user_email'],
     *      'conditions' => 'users_alias.ID = main_table.user_id',
    ])
     */

    protected function _check_conditions_matchings($conditions) {
        if (empty($conditions)) {
            return $conditions;
        }
        $matchings = ($this->_table->relations_by('source', 'matching'));
        $elected_matchings = [];

        foreach($matchings as $matching_key => $matching_value) {
        }

    }

    private function _compile_part($type) {

        //Glb_Log::info('$type ' . $type);
        switch ($type) {

            case 'relations' :
                // inform Glb_Table that a relation is loaded
                $this->_table->compile_relations($this->_get('relations', []), $this->_options);
                $this->_table->compile_joins($this->_get('join', []), $this->_options);
                break;

            case 'modifiers':
                // no one cares but modifiers are an option
                $this->_cache('modifiers', strtoupper(implode(' ', $this->_get('modifiers', []))));
                break;

            case 'from':
                if ($this->_type == 'select') {
                    $this->_cache('from', self::quote_field($this->_table->sql_name) . ' AS ' . self::quote_field($this->_table->sql_alias));
                } else {
                    $this->_cache('from', self::quote_field($this->_table->sql_name));
                }
                break;

            case 'table':
                $this->_cache('table', self::quote_field($this->_table->sql_name));
                break;

            case 'alias':
                $this->_cache('alias', self::quote_field($this->_table->sql_name) . ' AS ' . self::quote_field($this->_table->sql_alias));
                break;

            case 'where': case 'having':

                // check if a condition has not been
                // $condition_key = $this->_check_matching($condition_key);
                // $condition_value = $this->_check_matching($condition_value);
                //glb_dump($this->_get('relations'));

                $this->_cache($type, $this->_build_conditions($this->_get($type)));
                break;

            case 'group':
                $groups = [];
                foreach($this->_get('group', []) as $group) {
                    if (!is_array($group)) {
                        $groups[] = Glb_Db::instance()->escape($group);
                    } else {
                        foreach($group as $group_item) {
                            if (!is_array($group_item)) {
                                $groups[] = Glb_Db::instance()->escape($group_item);
                            }
                        }
                    }
                }
                $this->_cache('group', implode(', ', $groups));
                break;

            case 'order':
                $orders = [];
                $this->_set('order', Glb_Array::convert_associative($this->_get('order', []), 'asc'), true);

                foreach($this->_get('order', []) as $order_key => $order_value) {
                    if (in_array(strtolower($order_value), ['asc', 'desc'])) {
                        $orders[] = $this->_escape_inside_double_quotes($order_key) . ' ' . Glb_Db::instance()->escape(strtoupper($order_value));
                    }
                }
                $this->_cache('order', implode(', ', $orders));
                break;

            case 'into':
                if ($this->_type == 'insert') {
                    $this->_set('into', $this->_table->sql_name, true, true);
                }
                $this->_cache($type, $this->_get($type, '', 'last'));
                break;

            case 'limit': case 'offset':
                $this->_cache($type, $this->_get($type, '', 'last'));
                break;

            case 'set':
                // for insert, normalize 'fields' / 'values'...
                // first case, the programmer filled only 'values'
                // -> it means 'values' is an associative array (or... the programmer is an idiot)
                $fields = $this->_get('fields', null);
                $values = $this->_get('values', null);
                if ( $values === null || ($fields === null && !Glb_Array::is_associative($values)) ) {
                    // let him know the truth
                    throw new Exception(__CLASS__ . ': ' . __glb('Glb_Query: For an INSERT, you need to set "fields" AND "values", or "values" as an associative array.'));
                }

                if ($fields === null) {
                    $fields = array_keys($values);
                }

                $set = $this->_build_update_values($values, $fields);

                $this->_cache('set', $set);
                // @todo : check max 'max_allowed_packet'
                break;

            case 'fields':
                // a good practice
                if ($this->_type == 'insert') {
                    // for insert, normalize 'fields' / 'values'...
                    // first case, the programmer filled only 'values'
                    // -> it means 'values' is an associative array (or... the programmer is an idiot)
                    $fields = $this->_get('fields', null);
                    $values = $this->_get('values', null);
                    if ( $values === null ) {
                        // let him know the truth
                        throw new Exception(__CLASS__ . ': ' . __glb('Glb_Query: For an INSERT, you need to set "fields" AND "values", or "values" as an associative array.'));
                    } else if ($fields === null && (
                            ( ($level = Glb_Array::nested_level($values)) == 1 && !Glb_Array::is_associative($values) ) ||
                            ( $level > 1 && !Glb_Array::is_associative($values[0]) )
                        )) {
                            throw new Exception(__CLASS__ . ': ' . __glb('Glb_Query: For an INSERT, you need to set "fields" AND "values", or "values" as an associative array.'));
                        }

                    if ($fields === null) {
                        $level = Glb_Array::nested_level($values);
                        if ($level == 1) {
                            $fields = array_keys($values);
                        } else {
                            $fields = array_keys($values[0]);
                        }
                    }

                    $fields = array_map(function($field) { return self::quote_field($field); }, $fields);

                    $this->_cache('fields', '(' . implode(', ', $fields) . ')');
                    $this->_cache('values', '(' . implode('), (', $this->_build_insert_values($values, $fields)) . ')');
                    // @todo : check max 'max_allowed_packet'


                } else if ($this->_type == 'select') {

                    $compiled = $this->_table->compile_fields($this->_get('fields', []));
                    $fields = [];
                    foreach($compiled as $field_alias => $field) {
                        if (!empty($field['formula'])) {
                            $fields[] = $field['formula'] . ' AS ' . self::quote_field($field_alias);
                        } else {
                            $fields[] = self::quote_field($field['table_alias']) . '.' . self::quote_field($field['column_alias']) . ' AS ' .  self::quote_field($field_alias);
                        }
                    }
                    $this->_cache('fields', implode(', ', $fields));

                }
                break;

            case 'join':
                $eager_relations = $this->_table->relations_by('strategy', 'eager', true, true, 'flat');

                //$contain_relations = $this->_table->relations_by('type', ['inner', 'left'], true, true, 'flat');
                $this->_cache('relations', $eager_relations);

                $this->_debug('$contain_relations', $eager_relations);
                $tmp_items = [];
                foreach($eager_relations as $alias => $relation) {

                    if ($relation->source == 'not_matching') {
                        $primary_key = $relation->related_table()->primary_key();
                        $not_matching_condition = [];
                        foreach($primary_key as $column) {
                            $not_matching_condition[$relation->alias . '.' . $column['column_name'] . ' IS'] = null;
                        }
                        $this->_set('where', $not_matching_condition);
                    }

                    if ($relation->source == 'matching') {
                        $primary_key = $relation->related_table()->primary_key();
                        $matching_condition = [];
                        foreach($primary_key as $column) {
                            $matching_condition[$relation->alias . '.' . $column['column_name'] . ' IS NOT'] = null;
                        }
                        $this->_set('where', $matching_condition);
                    }

                    if (in_array($relation->nature, ['many_to_one', 'one_to_one', 'one_to_many'])) {

                        $join_str = strtoupper($relation->type) . " JOIN " . self::quote_field($relation->related_table()->sql_name) . ' AS ' .
                            self::quote_field($relation->alias) . ' ON (' .
                            self::quote_field($relation->primary_table()->sql_alias) . '.' . self::quote_field($relation->binding_key) . ' = ' .
                            self::quote_field($relation->alias) . '.' . self::quote_field($relation->foreign_key);

                        //Glb_Log::info('$relation->conditions', $relation->conditions);
                        if ($relation->has_conditions()) {
                            $conditions = $relation->conditions();
                            $conditions = Glb_Array::dedupe($conditions);
                            $join_str .= ' AND ' . $this->_build_conditions($conditions);
                        }
                        $tmp_items[] = $join_str . ')';

                    } else {

                        // first through join
                        $join_str = strtoupper($relation->type) . " JOIN " . self::quote_field($relation->through_table()->sql_name) . ' AS ' .
                            Glb_Query::quote_field($relation->through_table()->sql_alias) . ' ON (' .
                            self::quote_field($relation->through_table()->sql_alias) . '.' . self::quote_field(array_values($relation->binding_key)[0]) . ' = ' .
                            self::quote_field($relation->primary_table()->sql_alias) . '.' . self::quote_field(array_keys($relation->binding_key)[0]) . ') ' .
                            strtoupper($relation->type) . " JOIN " . self::quote_field($relation->related_table()->sql_name) . ' AS ' .
                            Glb_Query::quote_field($relation->related_table()->sql_alias) . ' ON (' .
                            self::quote_field($relation->related_table()->sql_alias) . '.' . self::quote_field(array_values($relation->foreign_key)[0]) . ' = ' .
                            self::quote_field($relation->through_table()->sql_alias) . '.' . self::quote_field(array_keys($relation->foreign_key)[0]);

                        // raise callback

                        //Glb_Log::info('$relation->conditions', $relation->conditions);
                        if ($relation->has_conditions()) {
                            $conditions = $relation->conditions();
                            $conditions = Glb_Array::dedupe($conditions);
                            $join_str .= ' AND ' . $this->_build_conditions($conditions);
                        }

                        $tmp_items[] = $join_str . ')';
                    }
                }

                $join_relations = $this->_table->joins();
                //glb_dump($join_relations);

                //$this->_cache('join', $join_relations);

                foreach($join_relations as $alias => $join) {
                    //Glb_Log::info('$join ', $join);
                    $join_str = strtoupper($join['type']) . " JOIN " . self::quote_field($join['table_object']->sql_name) . ' AS ' .
                        $join['alias'] . ' ON (' .
                        $this->_build_conditions($join['conditions']) . ')';
                    $tmp_items[] = $join_str;
                }
                $this->_cache('join', implode(" ", $tmp_items));
                break;

        }

    }
/*
    protected function _normalize_join($join) {
        if (empty($join['table'])) {
            throw new Exception('Glb_Query : Join ' . Glb_Text::quote_label('table') . ' attribute is required. ' . print_r($join, true));
        }
        $join = Glb_Hash::ensure_values($join, ['type', 'alias', 'conditions'], ['inner', $join['table'], []]);
        $join['type'] = strtolower(trim($join['type']));
        if (!in_array($join['type'], ['inner', 'left', 'right'])) {
            throw new Exception('Glb_Query : Join ' . Glb_Text::quote_label('type') . ' attribute should be LEFT, INNER or RIGHT. ' . Glb_Text::quote_label($join['type']) . ' given. '  . print_r($join, true));
        }
        if (empty($join['conditions'])) {
            throw new Exception('Glb_Query : Join ' . Glb_Text::quote_label('conditions') . ' attribute is required. ' . print_r($join, true));
        }
        return $join;
    }*/

    // compile query
    protected function _compile() {

        $this->_compile_init();

        $sql = $this->_templates['queries'][$this->_type];
        $parts = Glb_Text::between_all($sql, '{', '}');
        //Glb_Log::info('$sql ', $sql);

        $this->_compile_part('relations');
        $this->_debug($this->_data);

        foreach($parts as $part) {
            $this->_compile_part($part);

            //Glb_log::info('$part ', $part);
            //Glb_log::info('$partValue ', $this->_cache($part));
            //Glb_log::info('template ', $this->_templates['statements'][$part]);
            //Glb_log::info('replace ', sprintf($this->_templates['statements'][$part], $this->_cache($part)));
            if (empty($this->_cache($part)) && strpos($this->_templates['statements'][$part], ' %s') !== false) {
                $sql = str_replace('{' . $part . '}', '', $sql);
            } else {
                $sql = str_replace('{' . $part . '}', sprintf($this->_templates['statements'][$part], $this->_cache($part)), $sql);
            }
            //Glb_Log::info('$sql ', $sql);
        }

        return $sql;
    }

    public static function quote_field($field) {
        return Glb_Text::ensure_trailing(Glb_Text::ensure_leading($field, '`'), '`');
    }

    private function _escape_inside_double_quotes($string) {
        $parts = explode('"', $string);
        $parts = array_map(function($item) { return Glb_Db::instance()->escape($item); }, $parts);
        return implode('"', $parts);
    }

    public static function alias_field($table_alias, $column, $alias_delimiter = null) {
        return $table_alias . (empty($alias_delimiter) ? '___' : $alias_delimiter) . $column;
    }

    public static function unalias_field($field_alias, $alias_delimiter = null) {
        return explode((empty($alias_delimiter) ? '___' : $alias_delimiter), $field_alias, 2);
    }

    public static function parse_field($field) {
        $parts = Glb_Array::explode_nudge($field, '.', 2, '');
        $parts[0] = str_replace('`', '', $parts[0]);
        $parts[1] = str_replace('`', '', $parts[1]);
        return $parts;
    }

    public function &all() {
        if (!$this->_executed) {
            //glb_dump('execution');
            $this->execute();
        } else {
            //glb_dump('no execution');
        }
        return $this->__cache['results'];
    }

    /**
     * Get first $count elements returned by sql query
     * @param int|null $count If empty, then 1 is used
     * @return Glb_Entity|null
     */
    public function first($count = 1) {
        // read old "limit", then set it to 1 during the query
        // and restore after execution for later use of Glb_Query::all.
        $count = Glb_Validator::check_value($count, 'empty', 1);
        $changed = false;
        $old = $this->_get('limit', null, 'last');
        $this->_set('limit', $count, true, true);;
        $this->execute();
        $this->_set('limit', $old, true, true);
        return $this->__cache['results']->first();
    }

    /**
     * Get last $count elements returned by sql query
     * @param int|null $count If empty, then 1 is used
     * @return Glb_Entity|null
     */
    public function last($count = 1) {

        $old_orders = Glb_Array::convert_associative($this->_get('order', []), 'asc');
        $old_limit = $this->_get('limit', null, 'last');
        $new_orders = [];

        if (!empty($old_orders)) {
            // revert the order by clause
            foreach($old_orders as $key => $order) {
                if (strtolower($order) == 'desc') {
                    $new_orders[$key] = 'asc';
                } else {
                    $new_orders[$key] = 'desc';
                }
            }
            $this->_set('order', $new_orders, true);
            $this->_set('limit', 1, true);
            $this->execute();
            $this->_set('order', $old_orders);
            $this->limit($old_limit);
            return $this->__cache['results']->first();
        } else {
            // here the is nothing to do...
            // using "PRIMARY KEY DESC "is not a good practice because MySql lets his storage engine
            // ordering his results as it wants and this is rarely purely by PRIMARY ASC.
            // So... play the full query and return last result
            $this->execute();
            return $this->__cache['results']->last();
        }
    }

    /*public function execute_count() {
        $this->modifiers('SQL_CALC_FOUND_ROWS');
        $result = $this->execute();
        $rows = Glb_Db::instance()->raw_query('SELECT FOUND_ROWS()', []);
        $count = empty($rows[0]['FOUND_ROWS()']) ? 0 : $rows[0]['FOUND_ROWS()'];
        $this->_cache('count', $count);
        return $result;
    }*/

    public function count($refresh = false) {

        if (!empty($this->_cache('count')) && $this->_executed && !$refresh ) {
            // to avoid executing twice then same query
            return $this->_cache('count');
        }

        if (empty($this->_data['group']) && empty($this->_data['offset'])) {

            // if no 'group' and no 'offset', then we can simply replace fields by count(*)
            $old_fields = $this->_get('fields');
            $this->_set('fields', 'count(*)', true, true);
            $this->_build_sql();
            if ($this->is_stopped()) { return null; }

            // Run sql
            $rows = Glb_Db::instance()->raw_query($this->_cache('sql'), []);
            $count = empty($rows[0]['COUNT(*)']) ? 0 : $rows[0]['COUNT(*)'];
            $this->_set('fields', $old_fields, true, true);
            $this->_cache('count', $count);

        } else {

            // otherwise, we need to apply SQL_CALC_FOUND_ROWS and execute the full query
            $old_modifiers = $this->_get('modifiers');
            $this->_set('modifiers', 'SQL_CALC_FOUND_ROWS', false, true);
            $this->_build_sql();
            if ($this->is_stopped()) { return null; }

            // Run sql
            $rows = Glb_Db::instance()->raw_query($this->_cache('sql'), []);
            $rows = Glb_Db::instance()->raw_query('SELECT FOUND_ROWS()', []);
            $count = empty($rows[0]['FOUND_ROWS()']) ? 0 : $rows[0]['FOUND_ROWS()'];
            $this->_cache('count', $count);
            $this->_set('modifiers', $old_modifiers, true, true);

        }

        // return result
        return $this->_cache('count');
    }

    /*protected $_parts = [
        'select' => ['select', 'modifiers', 'from', 'join', 'where', 'group', 'having', 'order', 'limit', 'offset', 'union', 'epilog'],
        'delete' => ['delete', 'modifiers', 'from', 'where', 'epilog'],
        'update' => ['update', 'modifiers', 'set', 'join', 'where', 'epilog'],
        'insert' => ['insert', 'modifiers', 'values', 'epilog'],
    ];*/


    /**
     * The list of query clauses to traverse for generating a SELECT statement
     *
     * @var array
     */
    /*protected $_selectParts = [
        'select', 'from', 'join', 'where', 'group', 'having', 'order', 'limit',
        'offset', 'union', 'epilog'
    ];*/

    /*protected $_updateParts = ['update', 'set', 'where', 'epilog'];*/

    /**
     * The list of query clauses to traverse for generating a DELETE statement
     *
     * @var array
     */
    /*protected $_deleteParts = ['delete', 'modifier', 'from', 'where', 'epilog'];*/

    /**
     * The list of query clauses to traverse for generating an INSERT statement
     *
     * @var array
     */
    /*protected $_insertParts = ['insert', 'values', 'epilog'];*/

}

