<?php

/*
 * Glb_Db class. Wrapper helper for $wpdb
 */
final class Glb_Db
{

    private $_schema;
    private $_variables;
    private static $_schema_cache_settings = [];
    private static $_variables_cache_settings = [];

    public static function configure($cache_settings) {
        self::$_schema_cache_settings = $cache_settings['schema'];
        self::$_variables_cache_settings = $cache_settings['variables'];
    }

    /**
     * @function instance() : singleton function to get the unique instance loaded for the app
     * @return Glb_Db|null
     */
    public static function instance() {
        static $glb_db_instance = null;
        if ($glb_db_instance === null) {
            $glb_db_instance = new Glb_Db();
        }
        return $glb_db_instance;
    }

    public function __construct() {
    }

    /**
     * @param $table
     * @param $columns
     */
    public function check_columns($table, $columns, $exit = true) {
        Glb_Array::ensure($columns);
        $table_columns = $this->schema()->columns($table);

        foreach($columns as $column) {
            if (!array_key_exists($column, $table_columns)) {
                if ($exit) {
                    Glb_Request::instance()->set_error(400, 'redirect:referer', true);
                }
                return false;
            }
        }
        return true;
    }

    /**
     * @param $table
     * @param $columns
     */
    public function check_order_directions($directions, $actions = 'redirect:referer', $exit = true) {
        Glb_Array::ensure($directions);
        foreach($directions as $direction) {
            if (!in_array(strtolower($direction), ['asc', 'desc'])) {
                Glb_Request::instance()->set_error(400, $actions, $exit);
                return false;
            }
        }
        return true;
    }

    public function check_order_by($table, $columns, $exit = true) {
        Glb_Array::ensure($columns);
        $this->check_columns($table, $columns, $exit);
    }

    public function escape($string) {
        return esc_sql($string);
    }

    /**
     * @function table_name Calculates the real table name with WP prefix
     * @param string $table : the table name
     * @return string : The table name with prefix
     * @usage :
     *      Glb_Db::instance()->table_name('glb_core_logs') returns wp_glb_core_logs;
     */
    public function table_name($table) {
        global $wpdb;
        return Glb_Text::ensure_leading($table, $wpdb->prefix);
    }

    /**
     * Calculates the table name without WP prefix
     * @param string $table : the table name
     * @return string : The table name with prefix
     * @usage :
     *      Glb_Db::instance()->table_name('glb_core_logs') returns wp_glb_core_logs;
     */
    public function table_name_revert($table) {
        global $wpdb;
        return Glb_Text::remove_leading($table, $wpdb->prefix);
    }

    /**
     * Calculates the glb table name to be used in code (without WP and Glb prefix)
     * @param string $table : the table name
     * @usage :
     *      Glb_Db::instance()->table_name_revert('wp_glb_core_logs') returns glb_core_logs;
     */
    /*public function table_name_revert($table) {
    }*/


    public function db_name() {
        global $wpdb;
        return $wpdb->dbname;
    }

     /**
     * Get the schema definition
     */
    public function schema() {
        if (empty($this->_schema)) {
            $this->_schema = new Glb_Schema($this->db_name(), self::$_schema_cache_settings);
        }
        return $this->_schema;
    }

    public function variables() {
        if (empty($this->_variables)) {
            $file = Glb_File_Cache::get('variables_' . $this->db_name(), self::$_variables_cache_settings);
            if ($file->is_empty()) {
                $rows = $this->raw_query('SHOW VARIABLES;', null);
                $this->_variables = [];
                foreach($rows as $row) {
                    $this->_variables[$row['Variable_name']] = $row['Value'];
                }
                $file->data($this->_variables);
            }
            $this->_variables = $file->data();
        }
        return $this->_variables;
    }

    protected function _load_variables() {
        global $wpdb;
        $this->db = Glb_Db::instance();
        $this->_cache = Glb_File_Cache::get('variables_' . $wpdb->dbname, $settings);
        if ($this->_cache->is_empty()) {
            $this->_tables = $this->_load_database($database);
            $this->_cache->data($this->_tables);
        }
        $this->_tables = $this->_cache->data();
    }

    protected function _load_database($database) {
        $sql = 'SHOW TABLES FROM ' . $database;
        $tables = Glb_Db::instance()->raw_query($sql, [], false, ARRAY_N);
        $result = [];
        foreach($tables as $table) {
            $revert = $this->db->table_name_revert($table[0]);
            $result[$revert] = [
                'name' => $revert,
                'real_name' => $table[0],
                'attributes' => []
            ];
            $sql = 'SHOW TABLE STATUS WHERE Name="' . $table[0] . '"';
            $table_infos = Glb_Db::instance()->raw_query($sql, [], false, ARRAY_A);
            foreach($table_infos as $table_info) {
                foreach ($table_info as $key => $value) {
                    $result[$revert]['attributes'][strtolower($key)] = $value;
                }
            }
            $result[$revert]['indexes'] = $this->_load_table_indexes($table[0]);
            $result[$revert]['columns'] = $this->_load_table_columns($table[0]);
        }
        return $result;
    }

    /**
     * @function table_exists : check if table exists
     * @param string $table : the table name
     * @return boolean : true if table exists in database
     * @usage :
     *      Glb_Db::instance()->table_exists('glb_core_logs') returns true | false;
     */
    public function table_exists($table) {
        global $wpdb;
        $table = $this->table_name($table);
        $query = $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $table ) );
        // @todo : $wpdb->esc_like and different types of escape

        if ( $wpdb->get_var( $query ) == $table ) {
            return true;
        }
    }


    /*
     * @todo : implement complex where conditions
     */
    /*protected function build_where($where, $operator = '') {
        $result = ['sql' => [], 'format' => []];
        foreach($where as $where_operator_or_field => $where_item) {
            if (in_array($where_operator_or_field, ['AND', 'OR'])) {
                $sub_result = $this->build_where($where_item, $where_operator_or_field);
                $result['sql'] .= $result['sql'] . $sub_result['sql'];
                $result['format'] = array_merge($result['format'], $sub_result['format']);
            } else {
                $type = Glb_String::get_between_remove($where_operator_or_field, '(', ')');
                $operator_split = explode(' ', $where_operator_or_field);
                if (count($operator_split)==1) {
                    $operator_split[] = '=';
                }
                if (empty($type) || $type == '%s') {
                    if (!Glb_String::starts_with($where_item, '"') && !Glb_String::starts_with($where_item, "'")) {
                        $where_item = '"' . $where_item;
                    }
                    if (!Glb_String::ends_with($where_item, '"') && !Glb_String::ends_with($where_item, "'")) {
                        $where_item = $where_item . '"';
                    }
                    $result['sql'] .= ' ' . join(' ', $operator_split) . ' ' . $where_item;
                }
                $result['format'][] = (empty($type) ? '%s': $type);
            }
        }

        return $result;
    }*/

    /**
     * @function update : update rows in a table
     * @param string $table : the table name
     * @param array $data : data to set
     * @param array $where_array : where conditions
     * @param array $format : data formats
     * @param array $where_format : where formats
     * @return int|false The number of rows updated, or false on error.
     * @usage :
     *      Glb_Db::instance()->update('glb_core_settings',
     *              ['value' => 'my_setting_value'],
     *              ['plugin' => 'glb_core', 'name' => 'my_setting_name'],
     *              ['%s'],
     *              ['%s', '%s']);
    */

    public function update($table, $data, $where_array = [], $format = null, $where_format = null) {
        if (empty($data)) { return false; }

        global $wpdb;
        $timer_key = self::timer_start();
        $result = $wpdb->update($this->table_name($table), $data, $where_array, $format, $where_format);
        /*if ($result === false) {
            self::log_error($wpdb);
        }*/
        self::timer_stop($timer_key);
        return $result;
    }

    /*
     * @function timer_start : log the begin of the query
     * @param string $key [optional] : if null, will be calculated
     * @return string : the key used
     */
    private function timer_start($key = null) {
        //Glb_Log::trace('Sql query : ' . $message);
        if ($key === null) {
            $key = 'ts_' . microtime(true);
        }
        Glb_Timers::start($key);
        return $key;
    }

    /*
     * @function timer_stop : log the end of the query and evaluate slow log
     */
    private function timer_stop($key) {
        $duration = Glb_Timers::stop($key);
        global $wpdb;
        Glb_Log::trace('Sql query processed in ' . $duration . ' ms : ' . $wpdb->last_query);
        if ($wpdb->last_query == 'SELECT * FROM wp_glb_docx_repos WHERE folder = \'\'') {
            Glb_Log::trace(print_r(Glb_Array::simplify(debug_backtrace(), ['file', 'line', 'function']), true));
        };
        // @todo glb : set this configurable
        if ($duration > 200) {
            Glb_Log::warning('Slow log (' . $duration . ' ms) alert for query : ' . $wpdb->last_query);
        }
        if ($wpdb->last_error) {
            Glb_Log::error('Sql exception (' . $wpdb->last_error . ') for query : ' . $wpdb->last_query);
        }
    }

    /**
     * @function delete : delete rows in a table
     * @param string $table : the table name
     * @param array $where_array : where conditions
     * @param array $where_format : where formats
     * @return int|false The number of rows updated, or false on error.
     * @usage :
     *      Glb_Db::instance()->delete('glb_core_settings',
     *              ['plugin' => 'glb_core', 'name' => 'my_setting_name'],
     *              ['%s', '%s']);
     */
    public function delete($table, $where_array = [], $where_format = null)
    {
        global $wpdb;
        $timer_key = self::timer_start();
        $result = $wpdb->delete($this->table_name($table), $where_array, $where_format);
        if ($result === false) {
            Glb_Log::error('Sql exception : ' . $wpdb->last_query . ' : ' . $wpdb->last_error);
        }
        self::timer_stop($timer_key);
    }

    /**
     * @function optimize : optimize a table
     * @param $table : table name
     * @return void.
     * @usage :
     *      Glb_Db::instance()->optimize('glb_core_settings');
     */
    public function optimize($table)
    {
        global $wpdb;
        $timer_key = self::timer_start();
        $result = $wpdb->query('OPTIMIZE TABLE ' . $this->table_name($table) . ';');
        if ($result === false) {
            Glb_Log::error('Sql exception : ' . $wpdb->last_query . ' : ' . $wpdb->last_error);
        }
        self::timer_stop($timer_key);
        //return $result;
    }

    /*
     * @function insert : insert a new single row in a table
     * @param $table : db table
     * @param array $data : fields with keys and values ['name' => 'name_value, 'description' => 'description value', 'active' => 0]
     * @param array $format : array telling the format for each parameter of $data ['%s', '%s', %s]
     * @result int|bool : id of the autoincrement if available or true if success / false if failure
     */
    public function insert($table, $data, $format = null)
    {
        if (empty($data)) { return false; }

        global $wpdb;
        $timer_key = self::timer_start();
        $result = $wpdb->insert($this->table_name($table), $data, $format);
        if ($result === false) {
            Glb_Log::error('Sql exception : ' . $wpdb->last_query . ' : ' . $wpdb->last_error);
        }
        self::timer_stop($timer_key);
        if ($wpdb->insert_id) {
            return $wpdb->insert_id;
        } else {
            return true;
        }
    }

    public function last_insert_id()
    {
        global $wpdb;
        if ($wpdb->insert_id) {
            return $wpdb->insert_id;
        } else {
            return false;
        }
    }

    public function last_error() {
        global $wpdb;
        return $wpdb->last_error;
    }

    public function last_warnings() {
        global $wpdb;
        $items = $wpdb->get_results( 'SHOW WARNINGS;' );
        $result = [];
        foreach($items as $item) {
            $result[] = $item->Message;
        }
        return $result;
    }

    public function last_errors() {
        global $wpdb;
        $items = $wpdb->get_results( 'SHOW ERRORS;' );
        $result = [];
        foreach($items as $item) {
            $result[] = $item->Message;
        }
        return $result;
    }

    /*
     * @function update_or_insert : update a single row in a table, or insert if not found
     * @param $table : db table
     * @param array $data : fields with keys and values ['name' => 'name_value, 'description' => 'description value', 'active' => 0]
     * @param array $where : where conditions
     * @param array $format : array telling the format for each parameter of $data ['%s', '%s', %s]
     * @param array $where_format : array telling the format for each parameter of $data ['%s', '%s', %s]
     * @result int|bool : id of the autoincrement if available or true if success / false if failure
     */
    public function update_or_insert($table, $data = [], $where = [], $format = null, $where_format = null)
    {

        $result = $this->update($table, $data, $where, $format, $where_format);
        if ($result) { $result = true; }

        if ( !$result ) {
            Glb_Log::trace("update failed -> insert");
            $result = $this->insert($table, array_merge($data, $where), $format);
        }
        /*if ($result !== false) {
            global $wpdb;
            if (!empty($wpdb->insert_id)) {
                $result = $this->select($table, ['id' => $result->insert_id], ['%d']);
            }
        }*/
        return $result;
    }

    public function select_first($table, $cols = [], $where_string = null, $where_values = null, $use_cache = true, $output = ARRAY_A)
    {
        $result = $this->select($table, $cols, $where_string, $where_values, $use_cache, $output);
        if (empty($result)) {
            return null;
        } else {
            return $result[0];
        }
    }

    public function exists($table, $where_string = null, $where_values = null, $use_cache = true)
    {
        $result = $this->select_first($table, [], $where_string, $where_values, $use_cache);
        return !empty($result);
    }


    public function prepare($where_string, $where_values = null) {
        global $wpdb;
        if ($where_values === null) {
            return $where_string;
        } else {
            return $wpdb->prepare($where_string, $where_values);
        }
    }

    public function rows_affected() {
        global $wpdb;
        return $wpdb->rows_affected;
    }

    /*
     * OBJECT - result will be output as a numerically indexed array of row objects.
     * OBJECT_K - result will be output as an associative array of row objects, using first column's values as keys (duplicates will be discarded).
     * ARRAY_A - result will be output as an numerically indexed array of associative arrays, using column names as keys.
     * ARRAY_N - result will be output as a numerically indexed array of numerically indexed arrays.
     */
    public function select($table, $cols = [], $where_string = null, $where_values = [], $use_cache = true, $output = ARRAY_A) {

        if (empty($cols)) {
            $cols = '*';
        } else if (is_array($cols)) {
            $cols = join(',', $cols);
        }

        if (!empty($where_values) && !is_array($where_values)) {
            $where_values = [$where_values];
        }

        $use_cache = (current_user_can('manage_options')) ? false : $use_cache;
        $cache_key = Glb_Cache::key(__FILE__, __FUNCTION__, $table, $cols, $where_string, $where_values, $output);

        if (!$use_cache || !Glb_Cache::exists($cache_key)) {
            global $wpdb;
            $table = $this->table_name($table);
            if (!empty($where_string) && !Glb_Text::starts_with(strtoupper(trim($where_string)), 'ORDER')) {
                $where_string = 'WHERE ' . $where_string;
            }
            $timer_key = self::timer_start();
            if (!empty($where_string)) {
                $prepared = $this->prepare("SELECT $cols FROM $table $where_string", $where_values);
            } else {
                $prepared = "SELECT $cols FROM $table";
            }
            $results = $wpdb->get_results($prepared , $output);
            self::timer_stop($timer_key);
            Glb_Cache::set($cache_key, $results);
        }
        return Glb_Cache::get($cache_key);


        if (!empty($where)) {
            $where_result = $this->build_where($where);
            //var_dump($where_result);
            //$where = " WHERE " . ;
        }

        $use_cache = (current_user_can('manage_options')) ? false : $use_cache;
        $cache_key = $table . '~~' . $cols . '~~' . $output . '~~' . $where;

        if (!$use_cache || empty($cache[$cache_key])) {
            global $wpdb;
            $table = $this->table_name($table);
            Glb_Log::notice("SELECT $cols FROM $table $where;");
            $cache[$cache_key] = $wpdb->get_results( "SELECT $cols FROM $table $where;", $output );
        }

        Glb_Log::trace($cache);
        return $cache[$cache_key];
    }

    /*
     * OBJECT - result will be output as a numerically indexed array of row objects.
     * OBJECT_K - result will be output as an associative array of row objects, using first column's values as keys (duplicates will be discarded).
     * ARRAY_A - result will be output as an numerically indexed array of associative arrays, using column names as keys.
     * ARRAY_N - result will be output as a numerically indexed array of numerically indexed arrays.
     */
    public function select_distinct($table, $col, $where_string = null, $where_values = null, $use_cache = true, $output = ARRAY_A) {

        $use_cache = (current_user_can('manage_options')) ? false : $use_cache;
        $cache_key = Glb_Cache::key(__FILE__, __FUNCTION__, $table, $col, $where_string, $where_values, $output);

        //$table . '~~' . $cols . '~~' . 'ARRAY_A' . '~~' . $where_string . '~~' . print_r($where_values, true);
        if (!$use_cache || !Glb_Cache::exists($cache_key)) {
            global $wpdb;
            $table = $this->table_name($table);
            if (!empty($where_string) && !Glb_Text::starts_with($where_string, 'ORDER')) {
                $where_string = 'WHERE ' . $where_string;
            }
            //var_dump($wpdb->prepare("SELECT DISTINCT $col FROM $table $where_string", $where_values), $output);
            $timer_key = self::timer_start();

            $prepared = $this->prepare("SELECT DISTINCT $col FROM $table $where_string", $where_values);
            $results = $wpdb->get_results( $prepared, $output);

            self::timer_stop($timer_key);
            $result = [];
            foreach($results as $key => $result_item) {
                $result[$key] = $result_item[$col];
            }
            Glb_Cache::set($cache_key, $result);
        }
        //var_dump($cache);
        return Glb_Cache::get($cache_key);

    }

    /*private function column_definition($table, $sql_def) {

        $result = [
            'label' => $sql_def->Field,
            'name' => $sql_def->Field,
            'nullable' => Glb_Text::extract_boolean($sql_def->Null, true),
            'default' => $sql_def->Default,
            'primary' => ($sql_def->Key == 'PRI')
        ];

        $type = $sql_def->Type;
        $signed = null;

        // find unsigned keyword
        if (strpos($type, ' unsigned') !== false) {
            $signed = 'unsigned';
            $type = str_replace(' unsigned', '', $type);
        }
        // find signed keyword
        if (strpos($type, ' signed') !== false) {
            $signed = 'signed';
            $type = str_replace(' signed', '', $type);
        }

        // find between parenthesis infos
        $type_extra = Glb_Text::remove_between($type, '(', ')');

        $type = trim($type);

        if (!array_key_exists($type, $this->field_types)) {
            throw new Exception(sprintf('MySql type not supported for database column [%s].%s : %s', $table, $sql_def->Field, $type));
        }

        $type_infos = $this->field_types[$type];

        // set min / max
        if (array_key_exists('min', $type_infos)) { $result['min'] = $type_infos['min']; }
        if (array_key_exists('max', $type_infos)) { $result['max'] = $type_infos['max']; }
        if (array_key_exists('storage', $type_infos)) { $result['min'] = -2 ^ (2 * $type_infos['storage']); $result['max'] = 2 ^ (2 * $type_infos['storage']) - 1; }

        // adjust with signed / unsigned info
        if (in_array($type, ['bit', 'boolean', 'tinyint', 'smallint', 'mediumint', 'int', 'integer', 'bigint', 'float', 'double', 'decimal', 'numeric'])) {
            if ($signed == 'unsigned' && array_key_exists('min', $type_infos) && array_key_exists('max', $type_infos)) {
                $result['max'] = -$result['min'] + $result['max'];
                $result['min'] = 0;
            }
        }

        if (in_array($type, ['char', 'varchar'])) {
            $result['min_length'] = 0;
            if (!empty($type_extra)) {
                $result['max_length'] = $type_extra;
            } else {
                $result['max_length'] = 255;
            }
        }

        $result['type'] = $type_infos['type'];
        // @todo : manage float / double type_extra

        return $result;

    }*/

    /*
     * @function table_columns : get the sql table columns definitions
     * @param $table : the name of the table
     * @return array of Glb_Column
     */
    /*public function table_columns($table) {

        $cache_key = Glb_Cache::key(__FILE__, __FUNCTION__, $table);
        if (!Glb_Cache::exists($cache_key)) {
            $table = $this->table_name($table);
            $cols_sql = "DESCRIBE $table";
            global $wpdb;
            $columns = $wpdb->get_results( $cols_sql );

            $result = [];
            foreach ( $columns as $column ) {
                // Build an array of Field names
                $column_definition = $this->column_definition($table, $column);
                $result[] = new Glb_Column($column_definition['name'], $column_definition['type'], $column_definition['label'], $column_definition);
            }
            Glb_Cache::set($cache_key, $result);
        }
        return Glb_Cache::get($cache_key);
    }*/

    private function normalize_where($where_string) {
        if (empty($where_string) || empty(trim($where_string))) {
            return '';
        }
        if (!Glb_Text::starts_with(trim($where_string), 'ORDER')) {
            return ' WHERE ' . $where_string;
        }
        return $where_string;
    }

    /*
    public function prepare($query, $args) {
        global $wpdb;
        return $wpdb->prepare($query, $args);
    }
    */

    public function raw_query($query, $args, $use_cache = true, $output = ARRAY_A)
    {
        global $wpdb;
        if (empty($args)) {
            $prepared = $query;
        } else {
            $prepared = $wpdb->prepare($query, $args);
        }

        $cache_key = Glb_Cache::key(__FILE__, __FUNCTION__, $prepared);
        if (!$use_cache || !Glb_Cache::exists($cache_key)) {
            $timer_key = self::timer_start();
            //$result = $wpdb->query();
            $result = $wpdb->get_results($prepared, $output);
            self::timer_stop($timer_key);

            /*if ($result === false) {
                self::log_error($wpdb);
            }*/
            Glb_Cache::set($cache_key, $result);
        }
        return Glb_Cache::get($cache_key);
    }

    public function select_count($table, $col = '*', $where_string = '', $where_values = [], $use_cache = true) {
        $table = $this->table_name($table);

        $cache_key = Glb_Cache::key(__FILE__, __FUNCTION__, $table, $col, $where_string, $where_values);
        //$cache_key = $this->cache_key(__FUNCTION__, $table, "count($col)", $where_string, $where_values);
        if (!$use_cache || !Glb_Cache::exists($cache_key)) {
            global $wpdb;
            Glb_Log::info('$where_string ', $where_string);
            Glb_Log::info('$where_values ', $where_values);
            $where_string  = $this->normalize_where($where_string);
            $timer_key = self::timer_start();
            if (empty($where_string)) {
                Glb_Cache::set($cache_key, $wpdb->get_var("SELECT COUNT($col) FROM $table"));
            } else {
                Glb_Cache::set($cache_key, $wpdb->get_var($this->prepare("SELECT COUNT($col) FROM $table $where_string", $where_values)));
            }
            self::timer_stop($timer_key);
        }
        return Glb_Cache::get($cache_key);
    }

        /*
     * OBJECT - result will be output as a numerically indexed array of row objects.
     * OBJECT_K - result will be output as an associative array of row objects, using first column's values as keys (duplicates will be discarded).
     * ARRAY_A - result will be output as an numerically indexed array of associative arrays, using column names as keys.
     * ARRAY_N - result will be output as a numerically indexed array of numerically indexed arrays.
     */
    public function select_val($table, $col, $where_string = '', $where_values = [], $use_cache = true) {
        // @todo : revoir la gestion du cache -> faire une seule classe globale
        static $cache = [];

        $cache_key = $table . '~~' . $col . '~~' . 'ARRAY_A' . '~~' . $where_string . '~~' . print_r($where_values, true);
        if (!$use_cache || empty($cache[$cache_key])) {
            global $wpdb;
            $table = $this->table_name($table);
            //var_dump($where_values);
            $timer_key = self::timer_start();
            $cache[$cache_key] = $wpdb->get_var( $this->prepare("SELECT $col FROM $table WHERE $where_string", $where_values));
            self::timer_stop($timer_key);
        }
        //var_dump($col);
        //var_dump($where_values);
        //var_dump($cache[$cache_key]);
        return $cache[$cache_key];

        if (!empty($where)) {
            $where = " WHERE " . $this->build_where($where);
        }
        Glb_Log::trace($where);

        $use_cache = (current_user_can('manage_options')) ? false : $use_cache;
        Glb_Log::trace("current_user_can('manage_options')) : " . current_user_can('manage_options'));
        Glb_Log::trace(" use_cache $use_cache ");
        Glb_Log::trace(" use_cache " . print_r($cache, true));

        $cache_key = $table . '~~' . $col . '~~' . 'ARRAY_A' . '~~' . $where;
        if (!$use_cache || empty($cache[$cache_key])) {
            global $wpdb;
            $table = $this->table_name($table);
            Glb_Log::trace("SELECT $col FROM $table $where;");
            $cache[$cache_key] = $wpdb->get_var( "SELECT $col FROM $table $where;");
        }

        Glb_Log::notice($cache);
        return $cache[$cache_key];
    }

    public function create_table($table, $fields_def, $table_def, $if_not_exists = true) {
        $sql = "CREATE TABLE " . (!empty($if_not_exists) ? "IF NOT EXISTS " : "") . $this->table_name($table) . " (" . $fields_def . ") $table_def;";
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    public function drop_table($table) {
        $sql = "DROP TABLE IF EXISTS " . $this->table_name($table) . ";";
        global $wpdb;
        $wpdb->query($sql);
    }

    public function __debugInfo() {
        return [];
    }
}